//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERAPLATFORM_WARPINGPROPERTIES_H)
    #define CANDERAPLATFORM_WARPINGPROPERTIES_H

#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <CanderaPlatform/Device/Common/Base/WarpMatrix.h>
#include <CanderaPlatform/Device/Common/Base/DisplayOrientation.h>

namespace Candera {

class Display;

/** @addtogroup CommonDevice
 *  @{
 */

/**
 *  @brief WarpingProperties is a wrapper class warping properties used by the
 *  display.
 */

class WarpingProperties {
    friend class Display;
    friend class GlWarpingManager;

    public:

        /**
         * Set the matrix used for display warping.
         * See Display::SetWarpMatrix.
         * @param warpMatrix    Matrix that is set to the display.
         * @param index    Index of the warp matrix
         */
        void SetWarpMatrix(WarpMatrix& warpMatrix, UInt index = 0);

        /**
         * Get the matrix used for display warping.
         * See Display::GetWarpMatrix.
         * @param index    Index of the warp matrix
         * @return  The matrix that was set to the display.
         */
        const WarpMatrix& GetWarpMatrix(UInt index = 0) const;
        WarpMatrix& GetWarpMatrix(UInt index = 0);

        /**
        * Reset warp matrix at index
        * @param index    Index of warp matrix.
        */
        void ResetWarpMatrix(UInt index);

        /**
        *  Sets a warp matrix weight.
        *  @param index of the warp matrix.
        *  @param weight of the warp matrix, passed as a uniform to the Shader.
        */
        void SetWarpMatrixWeight(UInt index, Float weight);

        /**
        *  Retrieves the weight of this warp matrix.
        *  @param index The index of the warp matrix.
        *  @return The weight of this warp matrix.
        */
        Float GetWarpMatrixWeight(UInt index) const;

        /**
         * Sets the bounding rectangle used to determine the area of the warping texture to use.
         * @param m_warpImageBounds The bounds of the warping texture to map to mesh.
         */
        void SetWarpImageBounds(Rectangle& warpImageBounds) { m_warpImageBounds = warpImageBounds; SetWarpMatrixDirtyFlag(true); }
        
        /**
         * Retrieves the bounds of the warping image to map.
         * @return The bounds of the warping image to map.
         */
        const Rectangle& GetWarpImageBounds() const { return m_warpImageBounds; }

        /**
         * Specifies the orientation of the display.
         * @param displayOrientation The orientation of the display.
         */
        void SetDisplayOrientation(DisplayOrientation::Enum displayOrientation) { m_displayOrientation = displayOrientation; }

        /**
         * Retrieves the orientation of the display.
         * @Return The orientation of the display.
         */
        DisplayOrientation::Enum GetDisplayOrientation() const { return m_displayOrientation; }

        /**
         * Enables or disables warping.
         * See Display::SetWarpingEnabled.
         * @return True if warping can be enabled at this time. False otherwise.
         */
        void SetWarpingEnabled(bool isEnabled);

        /**
         * Returns the value set by SetWarpingEnabled, or the default 'false'.
         * See Display::SetWarpingEnabled.
         * @return Whether warping is enabled or not.
         */
        bool IsWarpingEnabled() const { return m_isWarpingEnabled; }

        /**
         * Set the value of the warp matrix dirty flag.
         * See Display::SetWarpingEnabled.
         * @param flag True to use the matrix, false to ignore.
         */
        void SetWarpMatrixDirtyFlag(bool flag) { m_isWarpMatrixDirty = flag; }

        /**
         * Get the value of the warp matrix dirty flag.
         * @return True to use the matrix, false to ignore.
         */
        bool IsWarpMatrixDirty() const { return m_isWarpMatrixDirty; }

    private:
        // Private constructor to avoid having this class instantiated
        // outside the display;
        WarpingProperties();
        WarpingProperties(const WarpingProperties&);

        bool IsWarpingMatrixValid() const;
        Int GetActualWarpMatrixCount() const;
        const Float* GetActualWarpMatrixWeight();

        bool m_isWarpingEnabled;
        bool m_isWarpMatrixDirty;

        enum  { c_maxWarpMatrixCount = 8};

        WarpMatrix m_defaultMatrix;
        WarpMatrix m_warpMatrix[c_maxWarpMatrixCount];
        Float m_warpMatrixWeight[c_maxWarpMatrixCount];
        Float m_actualWarpMatrixWeight[c_maxWarpMatrixCount];

        Rectangle m_warpImageBounds;
        DisplayOrientation::Enum m_displayOrientation;
};

/** @}*/ //end of CommonDevice

}   // namespace Candera

#endif  // CANDERAPLATFORM_WARPINGPROPERTIES_H
