//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GL_RADIAL_GRADIENT_BRUSH_H)
#define CANDERA_GL_RADIAL_GRADIENT_BRUSH_H

#include <Candera/Engine2D/Effects/BrushEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Vector2Property.h>
#include <Candera/Engine2D/Property/ColorProperty.h>

namespace Candera {

    /**
     *  @brief Output a rectangle filled with a radial gradient effect between two colors.
     * 
     * The brush fills a rectangle of a defined Size() with generated radial gradient between a
     * CenterColor() and a RadialColor(). The GradientCenter() and the GradientMagnitude() define the
     * gradient layout.
     */
    class GlRadialGradientBrush : public BrushEffect2D {
        FEATSTD_TYPEDEF_BASE(BrushEffect2D);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(GlRadialGradientBrush);
        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Access the size of the fill area. This is a two component vector
         *  of which the first component represents the width and the second the height
         *  in pixels of the filled rectangle, in local space.
         *  @return A property that can be used to modify the rectangle size.
         */
        Vector2Property& Size() { return m_size; }

        /**
         *  Defines a point in texture coordinates where the 'center' color is fully opaque.
         * @return Property that can be used to modify the gradient center.
         */
        Vector2Property& GradientCenter() { return m_gradientCenter; }

        /**
         *  Defines the magnitude of the gradient effect. The gradient effect radiates from the center point for a radius
         * equal with the magnitude, in texture coordinate space. The radial gradient decreases the opacity of the center color,
         * and increases the one of the radial color, proportional with the distance from the center point. Areas placed further
         * then the set magnitude from the center point are filled with the radial color.
         * @return Property that can be used to modify the gradient magnitude.
         */
        FloatProperty& GradientMagnitude() { return m_gradientMagnitude; }

        /**
         *  Defines the center color.
         *  @return Property that can be used to modify the center color.
         */
        ColorProperty& CenterColor() { return m_centerColor; }

        /**
         *  Defines the radial color.
         *  @return Property that can be used to modify the radial color.
         */
        ColorProperty& RadialColor() { return m_radialColor; }

        // overrides Effect2D::GetBoundingRectangle
        virtual void GetBoundingRectangle(Rectangle& boundingRectangle) const override;

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const override;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::GlRadialGradientBrush, GlRadialGradientBrush, EFFECT2D_TYPE_BRUSH)
            CdaDescription("The brush generates a radial gradient between two colors. The size of the generated rectangle"
                " is equal to the effective scale factor of the Node2D onto which it is applied.")
            CdaEffect2DProperties()
                CdaEffect2DImportBaseClassProperties(BrushEffect2D, Candera::BrushEffect2D)
                
                CdaEffect2DProperty(Size, Vector2, m_size)
                    CdaDescription("The filled size (width and height).")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(GradientCenter, Vector2, m_gradientCenter)
                    CdaDescription("Defines a point in texture coordinates where the 'center' color is fully opaque.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(GradientMagnitude, Float, m_gradientMagnitude)
                    CdaDescription(" Defines the magnitude of the gradient effect. The gradient effect radiates from the center point for a radius"
                    " equal with the magnitude, in texture coordinate space. The radial gradient decreases the opacity of the center color,"
                    " and increases the one of the radial color, proportional with the distance from the center point. Areas placed further"
                    " then the set magnitude from the center point are filled with the radial color.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(CenterColor, Color, m_centerColor)
                    CdaDescription("Defines the center color.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(RadialColor, Color, m_radialColor)
                    CdaDescription("Defines the radial color.")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

        /**
         *  Activate the parameters used for gradient generation.
         *  @param node The effective alpha value of the Node2D is relevant.
         *  @param output The parameters are activated for this 2D context.
         */
        void Activate(const Node2D& node, ContextHandle2D output);

        /**
         *  Deactivate the color used for for gradient generation.
         *  @param output The parameters are deactivated for this 2D context.
         */
        void Deactivate(ContextHandle2D output) const;

        /**
         *  Render the gradient without setting the colors.
         *  @param transform    3x2 transform matrix.
         *  @param output       The 2D context handle.
         *  @param outputArea   The output area as a rectangle.
         */
        void RenderGradient(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea);

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        GlRadialGradientBrush();
        explicit GlRadialGradientBrush(const GlRadialGradientBrush& rhs);
        virtual ~GlRadialGradientBrush();
        GlRadialGradientBrush& operator = (const GlRadialGradientBrush& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea) override;

    private:
        Vector2Property m_size;
        Vector2Property m_gradientCenter;
        FloatProperty m_gradientMagnitude;
        ColorProperty m_centerColor;
        ColorProperty m_radialColor;
    };

}   // namespace Candera

#endif  // CANDERA_GL_RADIAL_GRADIENT_BRUSH_H
