//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GL_OUTLINE_EFFECT_H)
#define CANDERA_GL_OUTLINE_EFFECT_H

#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/System/Properties/Property.h>
#include <Candera/Engine2D/Property/ColorProperty.h>

namespace Candera {

    /**
     *    @brief    This effect outlines a texture that contains an alpha channel. The RGB data of the texture is ignored.
     */
    class GlOutlineEffect : public InPlaceEffect2D {
         FEATSTD_TYPEDEF_BASE(InPlaceEffect2D);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(GlOutlineEffect);
        CANDERA_RTTI_DECLARATION(Candera::GlOutlineEffect);

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Property for the outline width in pixels. The outline width shall not be greater than the width of the thinnest line in the image.
         *  @return Reference to the simple property containing the outline width.
         */
        UInt8Property& OutlineWidth() { return m_outlineWidth; }

        /**
         *  Property for the outline color. 
         *  @return Reference to the property containing the outline color.
         */
        ColorProperty& OutlineColor() { return m_outlineColor; }

        /**
        *  Property for the color factor.
        *  @return Reference to the property containing the color factor.
        */
        Int& ColorFactor() { return m_colorFactor; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const override;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::GlOutlineEffect, GlOutlineEffect, EFFECT2D_TYPE_INPLACE)

            CdaDescription("This effect adds an outline to an image depending on the data in the alpha channel of an image. The RGB data if the image is ignored. This effect is designed for the use with the text. Do not use with any text cache type other than BitmapCache!")
            CdaEffect2DProperties()

            CdaEffect2DProperty(OutlineWidth, UInt8, m_outlineWidth)
                    CdaDescription("Width of the outline. The width shall not exceed the with of the thinnest lines to outline. Do not use with any text cache type other than BitmapCache!")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(OutlineColor, Color, m_outlineColor)
                    CdaDescription("Set color for the outline. Do not use with any text cache type other than BitmapCache!")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

        /**
         *  Activate the parameters used for flipping.
         *  @param output The parameters are activated for this 2D context.
         */
         void ActivateFilter(ContextHandle2D output);

        /**
         *  Deactivate the parameters used for flipping.
         *  @param output The parameters are deactivated for this 2D context.
         */
        void DeactivateFilter(ContextHandle2D output) const;

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        GlOutlineEffect();
        explicit GlOutlineEffect(const GlOutlineEffect& rhs);
        virtual ~GlOutlineEffect();
        GlOutlineEffect& operator = (const GlOutlineEffect& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea) override;

    private:
        UInt8Property m_outlineWidth;
        ColorProperty m_outlineColor;
        static const Color s_defaultOutlineColor;
        
        Int m_colorFactor;
    };

}   // namespace Candera

#endif  // CANDERA_GL_OUTLINE_EFFECT_H

