//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlOutlineEffect.h"

#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/RenderDevice2DOver3D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera {
    CANDERA_RTTI_DEFINITION(GlOutlineEffect)


      const Color GlOutlineEffect::s_defaultOutlineColor(0.0F, 0.0F, 0.0F, 0.0F);

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
     GlOutlineEffect::GlOutlineEffect() :
        Base(),
        m_outlineWidth(1),
        m_colorFactor(0)
    {
    }

    GlOutlineEffect::GlOutlineEffect(const GlOutlineEffect& rhs) :
        Base(rhs),
        m_outlineWidth(rhs.m_outlineWidth.Get()),
        m_outlineColor(rhs.m_outlineColor.Get()),
        m_colorFactor(rhs.m_colorFactor)
    {

    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlOutlineEffect::~GlOutlineEffect()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlOutlineEffect::SharedPointer GlOutlineEffect::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

            GlOutlineEffect* brush = FEATSTD_NEW(GlOutlineEffect);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlOutlineEffect::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlOutlineEffect::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& /*node*/,
                                 ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "GlOutlineEffect"));
        
        bool result = Renderer2D::SetTransformationMatrix(
            output,
            RenderDevice2D::SourceSurface,
            transform);

        result = result && RenderDevice2D::SetSurface(
            output,
            RenderDevice2D::SourceSurface,
            input);

        result = result && RenderDevice2D::SetActiveArea(
            output,
            RenderDevice2D::SourceSurface,
            inputArea.GetLeft(),
            inputArea.GetTop(),
            inputArea.GetWidth(),
            inputArea.GetHeight());

        if (result) {
            ActivateFilter(output);
            result = Renderer2D::Blit(output);
            DeactivateFilter(output);
        }

        if (result) {
            Float left;
            Float top;
            Float width;
            Float height;
            static_cast<void>(RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height));
            outputArea.Set(left, top, width, height);
        }
    }

    /******************************************************************************
     *  Activate
     ******************************************************************************/
    void GlOutlineEffect::ActivateFilter(ContextHandle2D output)
    {
        //ContextHandle2D context, UInt8 filterSize, Color outlineColor, Vector2 outlinePixelDimensions, Vector2 outlineImageDimensions
        static_cast<void>(Internal::RenderDevice2DOver3D::SetOutlineParameters(output, m_outlineWidth, Color(m_outlineColor), m_colorFactor));
    }
    /******************************************************************************
     *  Deactivate
     ******************************************************************************/
    void GlOutlineEffect::DeactivateFilter(ContextHandle2D output) const
    {
        static_cast<void>(Internal::RenderDevice2DOver3D::SetOutlineParameters(output, 0, s_defaultOutlineColor, m_colorFactor));
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer GlOutlineEffect::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlOutlineEffect)(*this));
    }

}   // namespace Candera

