//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlLinearGradientBrush.h"

#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/RenderDevice2DOver3D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/Engine2D/Core/Camera2D.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(GlLinearGradientBrush, BrushEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    GlLinearGradientBrush::GlLinearGradientBrush() :
        Base(),
        m_size(Vector2(50.0F, 50.0F)),
        m_gradientCenter(Vector2(0.5F, 0.5F)),
        m_gradientDirection(Vector2(0.5F, 0.5F)),
        m_gradientMagnitude(0.5F),
        m_positiveColor(Color(0.0F, 0.0F, 0.0F, 1.0F)),
        m_negativeColor(Color(1.0F, 1.0F, 1.0F, 1.0F))
    {
    }

    GlLinearGradientBrush::GlLinearGradientBrush(const GlLinearGradientBrush& rhs) :
        Base(rhs),
        m_size(rhs.m_size.Get()),
        m_gradientCenter(rhs.m_gradientCenter.Get()),
        m_gradientDirection(rhs.m_gradientDirection.Get()),
        m_gradientMagnitude(rhs.m_gradientMagnitude.Get()),
        m_positiveColor(rhs.m_positiveColor.Get()),
        m_negativeColor(rhs.m_negativeColor.Get())
    {

    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlLinearGradientBrush::~GlLinearGradientBrush()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlLinearGradientBrush::SharedPointer GlLinearGradientBrush::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        GlLinearGradientBrush* brush = FEATSTD_NEW(GlLinearGradientBrush);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlLinearGradientBrush::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlLinearGradientBrush::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform, const Node2D& node,
                                 ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "GlLinearGradientBrush"));

        if (m_size() != Vector2(0.0F, 0.0F)) {
            Activate(node, output);
            RenderGradient(transform, output, outputArea);
            Deactivate(output);
        }
    }

    /******************************************************************************
     *  GetBoundingRectangle
     ******************************************************************************/
    void GlLinearGradientBrush::GetBoundingRectangle(Rectangle& boundingRectangle) const
    {
        boundingRectangle.SetPosition(0.0F, 0.0F);
        boundingRectangle.SetSize(m_size());
    }

    /******************************************************************************
     *  ActivateColor
     ******************************************************************************/
    void GlLinearGradientBrush::Activate(const Node2D& node, ContextHandle2D output)
    {

        Float cameraAlpha = 1.0F;
        const Camera2D* camera2D = Renderer2D::GetActiveCamera();
        if (0 != camera2D) {
            if (camera2D->IsCameraEffectiveAlphaEnabled()) {
                cameraAlpha = camera2D->GetEffectiveAlphaValue();
            }
        }
        static_cast<void>(RenderDevice2D::SetSurfaceConstColor(
            output,
            RenderDevice2D::SourceSurface,
            m_negativeColor().red,
            m_negativeColor().green,
            m_negativeColor().blue,
            m_negativeColor().alpha * node.GetEffectiveAlphaValue() * cameraAlpha));

        Color gradientColor(m_positiveColor);
        gradientColor[3] *= node.GetEffectiveAlphaValue() * cameraAlpha;

        static_cast<void>(Internal::RenderDevice2DOver3D::SetGradientColor(output, gradientColor));
        static_cast<void>(Internal::RenderDevice2DOver3D::SetGradientDirection(output, m_gradientDirection));
        static_cast<void>(Internal::RenderDevice2DOver3D::SetGradientCenter(output, m_gradientCenter));
        static_cast<void>(Internal::RenderDevice2DOver3D::SetGradientMagnitude(output, 1.0F / m_gradientMagnitude));
    }
    /******************************************************************************
     *  DeactivateColor
     ******************************************************************************/
    void GlLinearGradientBrush::Deactivate(ContextHandle2D output) const
    {
        static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.F, 1.F, 1.F, 1.F));
        static_cast<void>(Internal::RenderDevice2DOver3D::SetGradientColor(output, Color(0.0F, 0.0F, 0.0F, 1.0F)));
        static_cast<void>(Internal::RenderDevice2DOver3D::SetGradientDirection(output, Vector2()));
        static_cast<void>(Internal::RenderDevice2DOver3D::SetGradientCenter(output, Vector2()));
        static_cast<void>(Internal::RenderDevice2DOver3D::SetGradientMagnitude(output, 0.0F));
    }
    /******************************************************************************
     *  RenderSolidRectangle
     ******************************************************************************/
    void GlLinearGradientBrush::RenderGradient(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea)
    {
        bool result = Renderer2D::SetTransformationMatrix(
            output,
            RenderDevice2D::SourceSurface,
            transform);

        result = result && RenderDevice2D::SetSurface(
            output,
            RenderDevice2D::SourceSurface,
            0);

        result = result && RenderDevice2D::SetActiveArea(
            output,
            RenderDevice2D::SourceSurface,
            0.0F,
            0.0F,
            m_size().GetX(),
            m_size().GetY());

        result = result && Renderer2D::Blit(output);

        if (result) {
            Float left;
            Float top;
            Float width;
            Float height;
            if (RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height)) {
                outputArea = Rectangle(left, top, width, height);
            }
        }
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer GlLinearGradientBrush::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlLinearGradientBrush)(*this));
    }

}   // namespace Candera
