//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GL_FLIP_EFFECT_H)
#define CANDERA_GL_FLIP_EFFECT_H

#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/System/Properties/Property.h>

namespace Candera {

    typedef SimpleProperty<bool> SimpleBoolProperty;
    /**
     *    @brief    This effect flips a texture horizontal, vertical or both horizontal and vertical.
     */
    class GlFlipEffect : public InPlaceEffect2D {
         FEATSTD_TYPEDEF_BASE(InPlaceEffect2D);

    public:
        FEATSTD_TYPEDEF_SHARED_POINTER(GlFlipEffect);
        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Set horizontal flipping.
         *  @return Property that can be used to enable /disable flipping.
         */
        SimpleBoolProperty& FlipHorizontal() { return m_flipH; }

        /**
         *  Set vertical flipping.
         *  @return Property that can be used to enable /disable flipping.
         */
        SimpleBoolProperty& FlipVertical() { return m_flipV; }

        /**
         * Set this to true to evaluate the nodes effective layout direction. In case of a right to left layout direction
         * (due to culture setting or explicit configuration) the opposite of FlipH will be used as effective FlipH
         * (in case of FlipH set to true the effect will flip for left to right layout direction and not flip for right to left layout direction).
         */
        void SetUseNodeLayoutDirectionEnabled(bool enable);

        /**
         * Returns the setting to use the node layout direction in addition to FlipHorizontal.
         * @return true if the node layout direction has to be used in addition to FlipHorizontal.
         */
        bool IsUseNodeLayoutDirectionEnabled() const;

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const override;


        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::GlFlipEffect, GlFlipEffect, EFFECT2D_TYPE_INPLACE)

            CdaDescription("This effect flips a texture horizontal, vertical or both horizontal and vertical.")
            CdaEffect2DProperties()

                CdaEffect2DProperty(FlipH, bool, m_flipH)
                    CdaDescription("Set horizontal flipping.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(FlipV, bool, m_flipV)
                    CdaDescription("Set vertical flipping.")
                CdaEffect2DPropertyEnd()

                CdaIndirectEffect2DProperty(UseNodeLayoutDirection, bool, IsUseNodeLayoutDirectionEnabled, SetUseNodeLayoutDirectionEnabled)
                    CdaDescription("Set this to true to evaluate the nodes effective layout direction. In case of a right to left layout direction (due to culture setting or explicit configuration) the opposite of FlipH will be used as effective FlipH (in case of FlipH set to true the effect will flip for left to right layout direction and not flip for right to left layout direction).")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

        /**
         *  Activate the parameters used for flipping.
         *  @param output The parameters are activated for this 2D context.
         */
        void ActivateFlip(ContextHandle2D output);

        /**
         *  Deactivate the parameters used for flipping.
         *  @param output The parameters are deactivated for this 2D context.
         */
        void DeactivateFlip(ContextHandle2D output) const;

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        GlFlipEffect();
        explicit GlFlipEffect(const GlFlipEffect& rhs);
        virtual ~GlFlipEffect();
        GlFlipEffect& operator = (const GlFlipEffect& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea) override;

    private:
        SimpleBoolProperty m_flipH;
        SimpleBoolProperty m_flipV;
    };

}   // namespace Candera

#endif  // CANDERA_GL_FLIP_EFFECT_H

