//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlDropShadowEffect.h"

#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/RenderDevice2DOver3D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera {
    CANDERA_RTTI_DEFINITION(GlDropShadowEffect)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
     GlDropShadowEffect::GlDropShadowEffect() :
        Base(),
        m_shadowEnabled(true),
        m_shadowColor(Color(0.9F, 0.9F, 0.9F, 0.9F)),
        m_angle(135),
        m_distance(3),
        m_scale(1),
        m_blurFilter(SoftShadow3x3),
        m_colorFactor(0)
    {
    }

    GlDropShadowEffect::GlDropShadowEffect(const GlDropShadowEffect& rhs) :
        Base(rhs),
        m_shadowEnabled(rhs.m_shadowEnabled.Get()),
        m_shadowColor(rhs.m_shadowColor.Get()),
        m_angle(rhs.m_angle.Get()),
        m_distance(rhs.m_distance.Get()),
        m_scale(rhs.m_scale.Get()),
        m_blurFilter(rhs.m_blurFilter.Get()),
        m_colorFactor(rhs.m_colorFactor)
    {

    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlDropShadowEffect::~GlDropShadowEffect()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlDropShadowEffect::SharedPointer GlDropShadowEffect::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        GlDropShadowEffect* brush = FEATSTD_NEW(GlDropShadowEffect);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlDropShadowEffect::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlDropShadowEffect::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& /*node*/,
                                 ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "GlDropShadowEffect"));
        
        bool result = Renderer2D::SetTransformationMatrix(
            output,
            RenderDevice2D::SourceSurface,
            transform);

        result = result && RenderDevice2D::SetSurface(
            output,
            RenderDevice2D::SourceSurface,
            input);

        result = result && RenderDevice2D::SetActiveArea(
            output,
            RenderDevice2D::SourceSurface,
            inputArea.GetLeft(),
            inputArea.GetTop(),
            inputArea.GetWidth(),
            inputArea.GetHeight());

        if (result) {
            if (m_shadowEnabled.Get()) {
                ActivateFilter(output);
                result = Renderer2D::Blit(output);
                DeactivateFilter(output);
                static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));
            }
            result = result && Renderer2D::Blit(output);
        }

        if (result) {
            Float left;
            Float top;
            Float width;
            Float height;
            static_cast<void>(RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height));
            outputArea.Set(left, top, width, height);
        }
    }

    /******************************************************************************
     *  Activate
     ******************************************************************************/
    void GlDropShadowEffect::ActivateFilter(ContextHandle2D output)
    {
        static_cast<void>(Internal::RenderDevice2DOver3D::SetDropShadowParameters(output, true, m_shadowColor.Get(), m_angle, m_distance, m_scale, m_colorFactor, (m_blurFilter.Get() != HardShadow)));
    }
    /******************************************************************************
     *  Deactivate
     ******************************************************************************/
    void GlDropShadowEffect::DeactivateFilter(ContextHandle2D output) const
    {
        static_cast<void>(Internal::RenderDevice2DOver3D::SetDropShadowParameters(output, false, m_shadowColor.Get(), m_angle, m_distance, m_scale, m_colorFactor, (m_blurFilter.Get() != HardShadow)));
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer GlDropShadowEffect::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlDropShadowEffect)(*this));
    }

    Rectangle GlDropShadowEffect::GetBorderSize() const
    {
        Float radians = Math::DegreeToRadian(180.0F - static_cast<Float>(m_angle));
        Vector2 offset(Math::Cosine(radians) * static_cast<Float>(m_distance), Math::Sine(radians) * static_cast<Float>(m_distance));
        Vector2 leftTop(Math::Minimum(offset.GetX() - static_cast<Float>(m_scale), 0.0F), Math::Minimum(offset.GetY() - static_cast<Float>(m_scale), 0.0F));
        Vector2 rightBottom(Math::Maximum(0.0F, offset.GetX() + static_cast<Float>(m_scale)), Math::Maximum(0.0F, offset.GetY() + static_cast<Float>(m_scale)));
        return Rectangle(leftTop, rightBottom - leftTop);
    }

}   // namespace Candera

