//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#include "Context2DOver3DDeviceProgram.h"

#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DDevicePool.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/ShaderDefinitionsFor2DOver3D.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/SurfaceAllocator2DOver3D.h>

#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageTrace.h>

#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>


namespace Candera {
    namespace Internal {

        const Candera::Char* Context2DOver3DDeviceProgram::s_uniformNames[UniformTypes::_LastEntry] =
        {
            "u_Transform",            //Transform,
            "u_Texture",              //Texture,
            "u_TexTransform",         //TextureTransform,
            "u_ConstTexColor",        //TextureColor,
            "u_Texture1",             //MaskTexture,
            "u_MaskTransform",        //MaskTransform,
            "u_constMaskColor",       //MaskColor,
            "u_ColorTransform",       //ColorTransform,
            "u_constGradientColor",   //GradientColor,
            "u_direction",            //GradientDirection,
            "u_magnitude",            //GradientMagnitude,
            "u_center",               //GradientCenter,
            "u_outlineColor",         //OutlineColor,
            "u_pixelDimension",       //OutlinePixelDimension,
            "u_outlineWidth",         //OutlineWidth,
            "u_shadowColor",          //DropShadowColor,
            "u_offset",               //DropShadowOffset,
            "u_scale",                //DropShadowScale
            "u_colorFactor"           //DropShadowColorFactor
        };

        static const Char* s_texture0ShaderParamName = "u_Texture";
        static const Char* s_texture1ShaderParamName = "u_Texture1";

        Context2DOver3DDeviceProgram::Context2DOver3DDeviceProgram(const void* vertexShader, const void* pixelShader, ProgramType programType, FeatStd::Internal::Guid const & guid) :
            m_vertexShader(vertexShader),
            m_pixelShader(pixelShader),
            m_programType(programType)
        {
            for (UInt32 contextIndex = 0; contextIndex < CANDERA_MAX_CONTEXT_COUNT; ++contextIndex) {
                for (UInt32 uniformIndex = 0; uniformIndex < UniformTypes::_LastEntry; ++uniformIndex) {
                    m_uniformLocation[uniformIndex][contextIndex] = -1;
                }
            }

            m_shader = Shader::Create();
            m_shader->SetGuid(guid);
            m_textureShaderParamNames[TextureName] = TextureShaderParamName(s_texture0ShaderParamName);
            m_textureShaderParamNames[TextureName1] = TextureShaderParamName(s_texture1ShaderParamName);
        }

        Context2DOver3DDeviceProgram::~Context2DOver3DDeviceProgram() {
            if (!Unload()) {
                CANDERA_DEVICE_LOG_WARN("Unload failed.\n");
            }
        }

        Int& Context2DOver3DDeviceProgram::UniformLocation(Int uniformType)
        {
            return m_uniformLocation[uniformType][ContextResourcePool::GetActive().GetIndex()];
        }

        bool Context2DOver3DDeviceProgram::Upload() {
            if (m_shader == 0) {
                return false;
            }
            if (m_shader->IsUploaded()) {
                return true;
            }
            if (!m_shader->SetVertexShader(m_vertexShader, 0)) {
                CANDERA_DEVICE_LOG_WARN("Shader attachment failed.\n");
                return false;
            }
            if (!m_shader->SetFragmentShader(m_pixelShader, 0)) {
                CANDERA_DEVICE_LOG_WARN("Shader attachment failed.\n");
                return false;
            }

            if (!m_shader->Upload()) {
                CANDERA_DEVICE_LOG_WARN("Shader upload failed.\n");
                return false;
            }

            for (UInt i = 0; i < UniformTypes::_LastEntry; ++i) {
                static_cast<void>(m_shader->GetUniformLocation(s_uniformNames[i], UniformLocation(i)));
            }

            return true;
        }

        bool Context2DOver3DDeviceProgram::Viewport(
            const Context2DOver3DState &context,
            const Context2DOver3DTarget &target,
            Rectangle &correction)
        {
            // Compute viewport
            Rectangle viewport;
            correction.SetLeft(0.F);
            correction.SetTop(0.F);

            Rectangle originalViewport(
                context.dstArea.GetLeft(),
                context.dstArea.GetTop(),
                context.dstArea.GetWidth() < 0.0F ? target.width : context.dstArea.GetWidth(),
                context.dstArea.GetHeight() < 0.0F ? target.height : context.dstArea.GetHeight());

            // Correct viewport to actual size if needed.
            if (!Math::FloatAlmostEqual(target.actualHeight, target.height)) {
                Float scale = target.actualHeight / target.height;
                originalViewport.SetTop(originalViewport.GetTop() * scale);
                originalViewport.SetHeight(originalViewport.GetHeight() * scale);
            }
            if (!Math::FloatAlmostEqual(target.actualWidth, target.width)) {
                Float scale = target.actualWidth / target.width;
                originalViewport.SetLeft(originalViewport.GetLeft() * scale);
                originalViewport.SetWidth(originalViewport.GetWidth() * scale);
            }

            viewport.SetLeft(Math::Floor(originalViewport.GetLeft()));
            viewport.SetWidth(Math::Ceil(originalViewport.GetWidth() + viewport.GetLeft() - originalViewport.GetLeft()));
            if (viewport.GetLeft() < 0.F) {
                correction.SetLeft(originalViewport.GetLeft());
                viewport.SetWidth(viewport.GetWidth() + viewport.GetLeft());
                viewport.SetLeft(0.F);
            }
            else {
                correction.SetLeft(originalViewport.GetLeft() - viewport.GetLeft());
            }
            if (viewport.GetWidth() + viewport.GetLeft() > target.actualWidth) {
                viewport.SetWidth(target.actualWidth - viewport.GetLeft());
            }
            correction.SetWidth(viewport.GetWidth());
            if (viewport.GetWidth() <= 0.0F) {
                return false;
            }

            viewport.SetTop(Math::Floor(originalViewport.GetTop()));
            viewport.SetHeight(Math::Ceil(originalViewport.GetHeight() + viewport.GetTop() - originalViewport.GetTop()));
            if (viewport.GetTop() < 0.F) {
                correction.SetTop(originalViewport.GetTop());
                viewport.SetHeight(viewport.GetHeight() + viewport.GetTop());
                viewport.SetTop(0.F);
            }
            else {
                correction.SetTop(originalViewport.GetTop() - viewport.GetTop());
            }
            if (viewport.GetHeight() + viewport.GetTop() > target.actualHeight) {
                viewport.SetHeight(target.actualHeight - viewport.GetTop());
            }
            correction.SetHeight(viewport.GetHeight());
            if (viewport.GetHeight() <= 0.0F) {
                return false;
            }

            viewport.SetWidth(viewport.GetWidth() / target.actualWidth);
            viewport.SetLeft(viewport.GetLeft() / target.actualWidth);
            viewport.SetHeight(viewport.GetHeight() / target.actualHeight);
            viewport.SetTop(viewport.GetTop() / target.actualHeight);

            RenderDevice::ActivateViewport(viewport, target.actualWidth, target.actualHeight);

            // Correct correction back to Candera pixel space.
            if (!Math::FloatAlmostEqual(target.actualHeight, target.height)) {
                Float scale = target.height / target.actualHeight;
                correction.SetTop(correction.GetTop() * scale);
                correction.SetHeight(correction.GetHeight() * scale);
            }
            if (!Math::FloatAlmostEqual(target.actualWidth, target.width)) {
                Float scale = target.width / target.actualWidth;
                correction.SetLeft(correction.GetLeft() * scale);
                correction.SetWidth(correction.GetWidth() * scale);
            }

            return true;
        }

        bool Context2DOver3DDeviceProgram::ActivateScissor(const Context2DOver3DState& context, const Context2DOver3DTarget& target)
        {
            bool rc = true;
            if (!context.isScissorEnabled) {
                rc = RenderDevice::ActivateRenderState(RenderDevice::ScissorTestEnable, 0);
            }
            else {
                const Rectangle& userScissorRectangle = context.userScissorRectangle;
                const Float renderTargetWidth = target.actualWidth;
                const Float renderTargetHeight = target.actualHeight;
                Rectangle scissor(userScissorRectangle.GetLeft() / renderTargetWidth, userScissorRectangle.GetTop() / renderTargetHeight,
                    userScissorRectangle.GetWidth() / renderTargetWidth, userScissorRectangle.GetHeight() / renderTargetHeight);

                static_cast<void>(RenderDevice::SetScissorRectangle(scissor, renderTargetWidth, renderTargetHeight));
                rc = RenderDevice::ActivateRenderState(RenderDevice::ScissorTestEnable, 1);
            }
            return rc;
        }

        bool Context2DOver3DDeviceProgram::Draw(
            const Context2DOver3DState &context,
            const Context2DOver3DTarget &target,
            MemoryManagement::SharedPointer<VertexBuffer> vertexBuffer,
            Rectangle &updatedArea)
        {
            // Set viewport
            Rectangle viewportCorrection;
            if (!Viewport(context, target, viewportCorrection)) {
                return true;
            }

            // Set shader.
            if (!ActivateShaderAndGeometry(vertexBuffer)) {
                return false;
            }

            //Store original 3D shader param names.
            m_textureShaderParamNames[TextureName].m_shaderParamName = ShaderParamNames::GetUniformName(ShaderParamNames::Texture);
            m_textureShaderParamNames[TextureName1].m_shaderParamName = ShaderParamNames::GetUniformName(ShaderParamNames::Texture1);
            //Change shader param names to use context shader param names.
            ShaderParamNames::SetUniformName(ShaderParamNames::Texture, m_textureShaderParamNames[TextureName].m_contextShaderParamName);
            ShaderParamNames::SetUniformName(ShaderParamNames::Texture1, m_textureShaderParamNames[TextureName1].m_contextShaderParamName);

            // Activate textures.
            UInt textureId = 0;
            if (!ActivateTexture(UniformLocation(UniformTypes::Texture), textureId++, context.srcTexture)) {
                return false;
            }

            Rectangle sourceArea;
            const Context2DOver3DDevicePool& devicePool = Context2DOver3DDevicePool::GetInstance();
            if (devicePool.HasCustomVertexBuffer()) {
                sourceArea = devicePool.GetCustomVertexBufferBoundingRectangle();
            }
            else {
                sourceArea = GetTextureRectangle(context.srcTexture);
            }

            Rectangle maskArea = GetTextureRectangle(context.mskTexture);
            Rectangle activeSourceArea(sourceArea);
            Rectangle activeMaskArea(maskArea);
            activeSourceArea.IntersectChild(context.srcArea);
            activeMaskArea.IntersectChild(context.srcArea);

            if (!ActivateTransform(UniformLocation(UniformTypes::Transform),
                activeSourceArea, viewportCorrection, context.srcMatrix, updatedArea)){
                return false;
            }
            if (!ActivateTextureTransform(UniformLocation(UniformTypes::TextureTransform),
                sourceArea, activeSourceArea, context.srcPackOrder, context.flipH, context.flipV)) {
                return false;
            }

            if (!ActivateColor(UniformLocation(UniformTypes::TextureColor), context.srcConstColor)) {
                return false;
            }

            if ((context.srcColorMatrixEnabled) && (!ActivateMatrix4(UniformLocation(UniformTypes::ColorTransform),
                context.srcColorMatrix))) {
                return false;
            }

            if (!ActivateRenderMode(context.renderMode)){
                return false;
            }

            if (!ActivateMaskTransform(activeSourceArea, activeMaskArea, context)) {
                return false;
            }
            if (!ActivateTexture(UniformLocation(UniformTypes::MaskTexture), textureId++, context.mskTexture)) {
                return false;
            }

            if (!ActivateScissor(context, target)) {
                return false;
            }

            if (IsGradientShader()) {
                if (!ActivateColor(UniformLocation(UniformTypes::GradientColor), context.gradientColor)) {
                    return false;
                }

                if (!ActivateVec2(UniformLocation(UniformTypes::GradientCenter), context.gradientCenter)) {
                    return false;
                }

                if (m_programType == CCCABlitLinearGradient) {
                    if (!ActivateVec2(UniformLocation(UniformTypes::GradientDirection), context.gradientDirection)) {
                        return false;
                    }
                }

                if (!ActivateFloat(UniformLocation(UniformTypes::GradientMagnitude), context.gradientMagnitude)) {
                    return false;
                }
            }

            if (IsOutlineShader()){

                if (!ActivateFloat(UniformLocation(UniformTypes::OutlineWidth), static_cast<Float>(context.outlineWidth))) {
                    return false;
                }
                if (!ActivateColor(UniformLocation(UniformTypes::OutlineColor), context.outlineColor)) {
                    return false;
                }
                if (!ActivateVec2(UniformLocation(UniformTypes::OutlinePixelDimension), context.outlinePixelDimensions)) {
                    return false;
                }
                if (!ActivateFloat(UniformLocation(UniformTypes::DropShadowColorFactor), static_cast<Float>(context.colorFactor))) {
                    return false;
                }
            }

            if (IsDropShadowShader()) {

                if (!ActivateColor(UniformLocation(UniformTypes::DropShadowColor), context.shadowColor)) {
                    return false;
                }
                if (!ActivateVec2(UniformLocation(UniformTypes::DropShadowOffset), (context.offset * Vector2(1.0F, -1.0F)))) {
                    return false;
                }
                if (!ActivateFloat(UniformLocation(UniformTypes::DropShadowScale), static_cast<Float>(context.scale))) {
                    return false;
                }
                if (!ActivateFloat(UniformLocation(UniformTypes::DropShadowColorFactor), static_cast<Float>(context.colorFactor))) {
                    return false;
                }
                if (!ActivateVec2(UniformLocation(UniformTypes::OutlinePixelDimension), context.outlinePixelDimensions)) {
                    return false;
                }
            }

            DrawArrays(vertexBuffer);

            //Restore original shader param names.
            ShaderParamNames::SetUniformName(ShaderParamNames::Texture, m_textureShaderParamNames[TextureName].m_shaderParamName);
            ShaderParamNames::SetUniformName(ShaderParamNames::Texture1, m_textureShaderParamNames[TextureName1].m_shaderParamName);

            return true;
        }

        bool Context2DOver3DDeviceProgram::Unload() {
            if ((m_shader == 0) || (!m_shader->IsUploaded())) {
                return true;
            }

            bool result = m_shader->Unload();

            for (UInt i = 0; i < UniformTypes::_LastEntry; ++i) {
                UniformLocation(i) = -1;
            }

            return result;
        }

        bool Context2DOver3DDeviceProgram::ActivateShaderAndGeometry(const MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer)
        {
            if ((m_shader == 0) || (vertexBuffer == 0)) {
                return false;
            }
            if (!m_shader->Activate()) {
                return false;
            }
            if (!vertexBuffer->Activate()) {
                return false;
            }
            return m_shader->BindAttributes(vertexBuffer, 0);
        }

        bool Context2DOver3DDeviceProgram::ActivateMaskTransform(const Rectangle& activeSourceArea, const Rectangle& activeMaskArea, const Context2DOver3DState& state)
        {
            if (UniformLocation(UniformTypes::MaskTransform) == -1) {
                return (state.mskTexture->GetTextureImage() == 0);
            }

            Float vertSign = 0.0F;
            Matrix3x2 matrix;

            switch (state.mskPackOrder) {
            case RenderDevice2D::DisplayPackOrder:
                vertSign = 1.F;
                break;
            case RenderDevice2D::VerticallyFlippedDisplayPackOrder:
                vertSign = -1.F;
                break;
            case RenderDevice2D::UnknownPackOrder:
            default:
                return false;
            }

            Matrix3x2 m1(state.srcMatrix);
            m1.Translate(activeSourceArea.GetLeft(), activeSourceArea.GetTop());
            m1.Scale(activeSourceArea.GetWidth(), activeSourceArea.GetHeight());

            Matrix3x2 m1inv(m1);
            m1inv.Inverse();

            Matrix3x2 m2(state.mskTrans);
            Matrix3x2 mx;

            m2.Translate(activeMaskArea.GetLeft(), activeMaskArea.GetTop());
            m2.Scale(activeMaskArea.GetWidth(), vertSign * activeMaskArea.GetHeight());

            mx = m2 * m1inv;

            mx.Translate(0.F, -(1.F - vertSign) * 0.5F);
            mx.Inverse();
            // calculation of mx can possibly be optimized to use inverse only once

            return ActivateMatrix(UniformLocation(UniformTypes::MaskTransform), mx);
        }

        bool Context2DOver3DDeviceProgram::ActivateTexture(Int textureLocation, UInt textureId, const MemoryManagement::SharedPointer<Texture>& texture)
        {
            if (texture == 0) {
                return false;
            }
            if (textureLocation == -1) {
                return (texture->GetTextureImage() == 0);
            }
            else {
                if (texture->GetTextureImage() == 0) {
                    return false;
                }
            }
            return texture->Activate(*m_shader, textureId);
        }

        Rectangle Context2DOver3DDeviceProgram::GetTextureRectangle(const MemoryManagement::SharedPointer<Texture>& texture) const
        {
            bool isImageAvailable =
                (texture != 0) &&
                (texture->GetTextureImage() != 0) &&
                (texture->GetTextureImage()->ToImageSource3D() != 0);
            if (isImageAvailable) {
                Float width = static_cast<Float>(
                    texture->GetTextureImage()->ToImageSource3D()->GetWidth());
                Float height = static_cast<Float>(
                    texture->GetTextureImage()->ToImageSource3D()->GetHeight());

                return Rectangle(0.F, 0.F, width, height);
            }
            else {
                return Rectangle(0.F, 0.F, -1.F, -1.F);
            }
        }

        bool Context2DOver3DDeviceProgram::ActivateTransform(
            Int transformLocation,
            const Rectangle& active,
            const Rectangle& correction,
            const Matrix3x2& transform,
            Rectangle& updated) const
        {
            if (transformLocation == -1) {
                return false;
            }
            Matrix3x2 matrix(transform);

            updated = active;
            updated.Transform(matrix);

            bool scale = true;
            const Context2DOver3DDevicePool& devicePool = Context2DOver3DDevicePool::GetInstance();
            if ((devicePool.HasCustomVertexBuffer()) && (!devicePool.IsCustomVertexBufferNormalized())) {
                scale = false;
            }

            if (scale) {
                matrix.Translate(active.GetLeft(), active.GetTop());
                matrix.Scale(active.GetWidth(), active.GetHeight());
            }

            // Bring viewport space to OpenGL viewport space.
            matrix.PreTranslate(correction.GetLeft(), correction.GetTop());
            matrix.PreScale(2.0F / correction.GetWidth(), -2.0F / correction.GetHeight());
            matrix.PreTranslate(-1.0F, 1.0F);

            return ActivateMatrix(transformLocation, matrix);
        }

        bool Context2DOver3DDeviceProgram::ActivateTextureTransform(
            Int transformLocation,
            const Rectangle& size,
            const Rectangle& active,
            RenderDevice2D::PackOrder packOrder,
            bool flipH, bool flipV) const
        {

            if (transformLocation == -1) {
                return (packOrder == RenderDevice2D::UnknownPackOrder);
            }

            Matrix3x2 matrix;
            Float vertSign = 0.F;
            Float horizSign = (flipH) ? -1.0F : 1.0F;

            switch (packOrder) {
            case RenderDevice2D::DisplayPackOrder:
                vertSign = 1.F;
                if (flipV) {
                    vertSign = -1.F;
                }
                break;
            case RenderDevice2D::VerticallyFlippedDisplayPackOrder:
                vertSign = -1.F;
                if (flipV) {
                    vertSign = 1.F;
                }
                break;
            case RenderDevice2D::UnknownPackOrder:
            default:
                return false;
            }

            bool normalized = true;
            const Context2DOver3DDevicePool& devicePool = Context2DOver3DDevicePool::GetInstance();
            if ((devicePool.HasCustomVertexBuffer()) && (!devicePool.IsCustomVertexBufferNormalized())) {
                normalized = false;
            }

            if (normalized) {

                matrix.Translate((1.F - horizSign) * 0.5F, (1.F - vertSign) * 0.5F);
                matrix.Scale(horizSign / size.GetWidth(), vertSign / size.GetHeight());

                matrix.Translate(active.GetLeft(), active.GetTop());
                matrix.Scale(active.GetWidth(), active.GetHeight());
            }
            else {
                //In case of Mesh2D (denormalized VertexBuffer) we want to rely on the set texture coordinates.
                matrix.SetIdentity();
            }

            return ActivateMatrix(transformLocation, matrix);
        }

        bool Context2DOver3DDeviceProgram::ActivateColor(Int colorLocation, const Color& color) const
        {
            if (colorLocation == -1) {
                return false;
            }
            return RenderDevice::SetUniform<Shader::FloatVec4>(colorLocation, &color.GetData());
        }


        bool Context2DOver3DDeviceProgram::ActivateMatrix(Int matrixLocation, const Matrix3x2& matrix) const
        {
            if (matrixLocation == -1) {
                return false;
            }
            FEATSTD_LINT_CURRENT_SCOPE(446, "Violates MISRA C++ 2008 Required Rule 6-5-3: only getter, no side effect")
                Float mat[4][4] = {
                    { matrix(0, 0), matrix(1, 0), 0.F, matrix(2, 0) },
                    { matrix(0, 1), matrix(1, 1), 0.F, matrix(2, 1) },
                    { 0.F, 0.F, 1.F, 0.F },
                    { 0.F, 0.F, 0.F, 1.F }
            };

            return RenderDevice::SetUniform<Shader::FloatMat4>(matrixLocation, mat);
        }

        bool Context2DOver3DDeviceProgram::ActivateMatrix4(Int matrixLocation, const Matrix4& matrix) const
        {
            if (matrixLocation == -1) {
                return false;
            }
            FEATSTD_LINT_CURRENT_SCOPE(446, "Violates MISRA C++ 2008 Required Rule 6-5-3: only getter, no side effect")
                Float mat[4][4] = {
                    { matrix(0, 0), matrix(1, 0), matrix(2, 0), matrix(3, 0) },
                    { matrix(0, 1), matrix(1, 1), matrix(2, 1), matrix(3, 1) },
                    { matrix(0, 2), matrix(1, 2), matrix(2, 2), matrix(3, 2) },
                    { matrix(0, 3), matrix(1, 3), matrix(2, 3), matrix(3, 3) }
            };

            return RenderDevice::SetUniform<Shader::FloatMat4>(matrixLocation, mat);
        }

        bool Context2DOver3DDeviceProgram::ActivateVec2(Int location, const Vector2& vector) const
        {
            if (location == -1) {
                return false;
            }

            return RenderDevice::SetUniform<Shader::FloatVec2>(location, vector.GetData());
        }

        bool Context2DOver3DDeviceProgram::ActivateFloat(Int location, const Float& val) const
        {
            if (location == -1) {
                return false;
            }

            return RenderDevice::SetUniform<Shader::Float>(location, &val);
        }

        bool Context2DOver3DDeviceProgram::ActivateRenderMode(
            const MemoryManagement::SharedPointer<RenderMode>& renderMode) const
        {
            if (renderMode == 0) {
                return false;
            }
            return renderMode->Activate();
        }

        void Context2DOver3DDeviceProgram::DrawArrays(const MemoryManagement::SharedPointer<VertexBuffer>& vertexBuffer) const
        {
            vertexBuffer->Render();
        }

    }
}
