//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CONTEXT_2D_OVER_3D_H)
#define CONTEXT_2D_OVER_3D_H

#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DDevicePool.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DState.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DTarget.h>

namespace Candera
{
    namespace Internal {

/**
 * @brief Class that handles the states used by the render device 2D.
 *
 * This class should not be used directly. It is used by the render device 2D,
 * which matches context handles to instances of this class.
 *
 * Internal render targets may access this class directly.
 * For External render targets use the interface from DevicePackageInterface:
 *      CreateRenderContext2D
 *      DestroyRenderContext2D
 *      UpdateRenderContext2D
 */
class Context2DOver3D{
public:
    /**
     * Constructor.
     */
    Context2DOver3D();
    /**
     * Destructor.
     */
    ~Context2DOver3D();

    /**
     * Upload all resources associated with this context.
     * @return True if successful.
     */
    bool Upload();
    /**
     * Unload all resources associated with this context.
     * @return True if successful.
     */
    bool Unload();

    /**
     * Draw to the content corresponding to the current state.
     * @return True if successful.
     */
    bool Draw();
    /**
     * Clear the destination rectangle to the given color.
     * @param color Color used for clearing the destination rectangle.
     * @return True if successful.
     */
    bool Clear(const Color& color);

    /**
     * Set the size of the target surface.
     * The specified size should be the one of the associated render target,
     * as it is used for setting the viewport transformation.
     * This size is also used for mapping the 2D Candera space.
     * @param width Width of the target.
     * @param height Height of the target.
     */
    void SetSize(Int width, Int height) { SetSize(width, height, width, height); }
    /**
     * Set the size of the target surface.
     * This allows specifying different values for the Candera 2D space
     * and the viewport transformation to the render target.
     * @param width Width of the Candera 2D space.
     * @param height Height of the Candera 2D space.
     * @param actualWidth Width of the render target.
     * @param actualHeight Height of the render target.
     */
    void SetSize(Int width, Int height, Int actualWidth, Int actualHeight);

    /**
     * Update linkage of the new render target. This needs to be called
     * after SetSize, if upload has already been called.
     * @return True if successful.
     */
    bool Update() const {
        return true;
    }

    /**
     * Set the program used by the context to a value that allows it to
     * handle the surface given as parameter.
     * @param surfaceHandle surface that will be used when drawing.
     */
    void ChooseProgram(SurfaceHandle surfaceHandle);

    /**
    * Set the geometry to be further used for drawing.
    * @param customGeometry Geometry that will be used when drawing.
    */
    void SetCustomGeometry(GeometryHandle customGeometry) { m_customGeometry = customGeometry; }

    /**
    * Gets the geometry to be further used for drawing.
    * @return Geometry that will be used when drawing.
    */
    GeometryHandle GetCustomGeometry() const { return m_customGeometry; }


    /**
     * Retrieve the Target object within this context.
     * @return Immutable reference to the target object.
     */
    const Context2DOver3DTarget& GetTarget() const { return m_target; }
    /**
     * Retrieve the State object within this context.
     * @return Reference to the state object.
     */
    const Context2DOver3DState& GetState() const { return m_state; }
    Context2DOver3DState& GetState() { return m_state; }

private:
    bool m_isInitialized;

    Context2DOver3DDeviceProgram::ProgramType m_programType;

    Context2DOver3DState m_state;
    Context2DOver3DState m_stateForClearing;
    Context2DOver3DTarget m_target;
    GeometryHandle m_customGeometry;
};

}} // Name spaces.

#endif //CONTEXT_2D_OVER_3D_H
