//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GRAPHIC_DEVICE_UNIT_OWNER_ACCESS_H)
#define GRAPHIC_DEVICE_UNIT_OWNER_ACCESS_H

#include <Candera/Environment.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>

namespace Candera {

    class GraphicDeviceUnit;
    class ContextResourcePool;
    class Camera;

    namespace Internal {

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief Class used for accessing the owner of the GraphicDeviceUnit.
 */
class GraphicDeviceUnitOwnerAccess {
public:
    enum OwnerType
    {
        NoOwner,                    ///< No Owner.
        GraphicDeviceUnitOwner,     ///< Graphic device unit owner.
        ContextResourcePoolOwner    ///< Context resource pool owner.
    };

    /**
     * Retrieve an object that allow access to a GraphicDeviceUnitOwner.
     * @param base Object from which the owner is retrieved.
     */
    template<typename Base>
    static GraphicDeviceUnitOwnerAccess GetOwnerAccess(Base& base);

    /**
     * Empty Constructor.
     */
    GraphicDeviceUnitOwnerAccess();

    /**
     * Constructor.
     * @param hostGdu       Graphic device unit from which the Owner property 
     *                      will be retrieved.
     */
    GraphicDeviceUnitOwnerAccess(GraphicDeviceUnit* hostGdu);

    /**
     * Retrieve the type of owner of this render target.
     * @return Type of the owner.
     */
    OwnerType GetOwnerType() const { return m_ownerType; }

    /**
     * Retrieve the graphic device unit which acts as a owner for the attached
     * GraphicDeviceUnit.
     * @return Pointer to a GraphicDeviceUnit or 0, if such owner is not available.
     */
    GraphicDeviceUnit* GetGraphicDeviceUnitOwner() const;

    /**
     * Set the render target which acts as a owner for the host GraphicDeviceUnit.
     * @param host Host GrpahicDeviceUnit to which the owner wil be set.
     * @param owner RenderTarget3D owner.
     */
    static void SetGraphicDeviceUnitOwner(GraphicDeviceUnit* host, RenderTarget3D* owner);


    /**
     * Set the render target which acts as a owner for the Base.
     * @param base Object that will receive the owner.
     * @param owner RenderTarget3D owner.
     */
    template<typename Base>
    static void SetOwner(Base& base, RenderTarget3D* owner);

    /**
     * Retrieve the context resource pool which acts as a owner for the attached
     * GraphicDeviceUnit.
     * @return Pointer to a ContextResourcePool or 0, if such owner is not available.
     */
    ContextResourcePool* GetContextResourcePoolOwner() const;

    /**
     * Retrieve the context resource pool which acts as a owner for the attached
     * GraphicDeviceUnit.
     * @return Pointer to a ContextResourcePool or 0, if such owner is not available.
     */
    Camera* GetCompositionCamera() const { return m_camera; }

protected:
    // Constructors used internally.
    GraphicDeviceUnitOwnerAccess(GraphicDeviceUnit* gdu, Camera* camera);
    GraphicDeviceUnitOwnerAccess(ContextResourcePool* crp, Camera* camera);

    OwnerType m_ownerType;
    union {
        GraphicDeviceUnit* m_gdu;
        ContextResourcePool* m_crp;
    };
    Camera* m_camera;
};

/**
 * @brief Class for casting an object to GraphicDeviceUnitOwnerAccess.
 */
template <typename Base, typename Owner>
class GraphicDeviceUnitOwnerCast
{
public:
    static Owner* GetOwner(Base& /*base*/) { return 0; }
    static bool SetOwner(Base& /*base*/, Owner* /*owner*/) { return true; }
};

template <typename Base>
class GraphicDeviceUnitOwnerCast<Base, typename Base::Owner>
{
public:
    static typename Base::Owner* GetOwner(Base& base) 
    {
        return base.GetOwner();
    }
    static void SetOwner(Base& base, typename Base::Owner* owner)
    {
        base.SetOwner(owner);
    }
};

template <typename Base>
class GraphicDeviceUnitOwnerCast<Base, typename Base::OwnerType>
{
public:
    static typename Base::OwnerType* GetOwner(Base& base) 
    {
        return base.Owner().Get();
    }
    static void SetOwner(Base& base, typename Base::OwnerType* owner)
    {
        base.Owner.Set(owner);
    }
};

template <typename Base>
class GraphicDeviceUnitOwnerCast<Base, typename Base::CompositionCameraType>
{
public:
    static typename Base::CompositionCameraType* GetOwner(Base& base) 
    {
        return base.CompositionCamera().Get().Get();
    }
    static void SetOwner(Base& base, typename Base::CompositionCameraType* owner)
    {
        base.CompositionCamera().Get().Set(owner);
    }
};

template<typename Base>
GraphicDeviceUnitOwnerAccess GraphicDeviceUnitOwnerAccess::GetOwnerAccess(Base& base)
{
    Camera* camera = GraphicDeviceUnitOwnerCast<Base, Camera>::GetOwner(base);
    GraphicDeviceUnit* gdu = GraphicDeviceUnitOwnerCast<Base, GraphicDeviceUnit>::GetOwner(base);
    if (gdu != 0) {
        return GraphicDeviceUnitOwnerAccess(gdu, camera);
    }
    RenderTarget3D* rt3 = GraphicDeviceUnitOwnerCast<Base, RenderTarget3D>::GetOwner(base);
    if ((rt3 != 0) && (rt3->GetGraphicDeviceUnit() != 0)) {
        return GraphicDeviceUnitOwnerAccess(rt3->GetGraphicDeviceUnit(), 0);
    }
    ContextResourcePool* crp = GraphicDeviceUnitOwnerCast<Base, ContextResourcePool>::GetOwner(base);
    if (crp != 0) {
        return GraphicDeviceUnitOwnerAccess(crp, camera);
    }
    return GraphicDeviceUnitOwnerAccess();
}

template<typename Base>
void GraphicDeviceUnitOwnerAccess::SetOwner(Base& base, RenderTarget3D* owner)
{
    GraphicDeviceUnitOwnerCast<Base, RenderTarget3D>::SetOwner(base, owner);
}


/** @}*/ //end of CommonDevice

    }
}

#endif // GRAPHIC_DEVICE_UNIT_OWNER_ACCESS_H
