//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlWarpingProgram.h"
#include <CanderaPlatform/Device/Common/OpenGLES/GlTrace.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/Internal/GL/GlWarpingShaderDefinition.h>

namespace Candera
{

    static const Char* s_warpShaderTable[][2] = {
        GL_WARPING_BUILD_SHADER(1),
        GL_WARPING_BUILD_SHADER(2),
        GL_WARPING_BUILD_SHADER(3),
        GL_WARPING_BUILD_SHADER(4),
        GL_WARPING_BUILD_SHADER(5),
        GL_WARPING_BUILD_SHADER(6),
        GL_WARPING_BUILD_SHADER(7),
        GL_WARPING_BUILD_SHADER(8)
    };

GlWarpingProgram::GlWarpingProgram() :
    m_displayOrientation(DisplayOrientation::Unchanged),
    m_textureUniformLocation(-1),
    m_invertUniformLocation(-1),
    m_warpWeightUniformLocation(-1)
{
    m_shader = Shader::Create();
    m_renderMode = RenderMode::Create();

    if (m_renderMode != 0) {
        m_renderMode->SetBlendingEnabled(false);
        m_renderMode->SetCulling(RenderMode::NoCulling);
        m_renderMode->SetDepthTestEnabled(false);
        m_renderMode->SetDepthBias(0.0F, 0.0F);
        m_renderMode->SetStencilTestEnabled(false);
    }
}

GlWarpingProgram::~GlWarpingProgram()
{
    Unload();
}

bool GlWarpingProgram::Upload(Int positionAttributeCount)
{
    if (m_shader == 0) {
        return false;
    }

    if (m_shader->IsUploaded()) {
        return true;
    }

    if (!m_shader->SetVertexShader(s_warpShaderTable[positionAttributeCount - 1][0], 0)) {
        CANDERA_DEVICE_LOG_WARN("Shader attachment failed for GlWarpingProgram.");
        return false;
    }

    if (!m_shader->SetFragmentShader(s_warpShaderTable[positionAttributeCount - 1][1], 0)) {
        CANDERA_DEVICE_LOG_WARN("Shader attachment failed for GlWarpingProgram.");
        return false;
    }

    if (!m_shader->Upload()) {
        CANDERA_DEVICE_LOG_WARN("Shader upload failed for GlWarpingProgram.");
        return false;
    }

    static_cast<void>(m_shader->GetUniformLocation("u_Texture", m_textureUniformLocation));
    static_cast<void>(m_shader->GetUniformLocation("u_Invert", m_invertUniformLocation));
    static_cast<void>(m_shader->GetUniformLocation("u_WarpWeight", m_warpWeightUniformLocation));

    return true;
}

void GlWarpingProgram::Unload()
{
    if (IsUploaded()) {
        if (!m_shader->Unload()) {
            CANDERA_DEVICE_LOG_WARN("Shader unload failed for GlWarpingProgram.");
        }
    }

    m_textureUniformLocation = -1;
    m_invertUniformLocation = -1;
    m_warpWeightUniformLocation = -1;

}

void GlWarpingProgram::Activate() const
{
    if (IsUploaded()) {

        if (!m_shader->Activate()) {
            CANDERA_DEVICE_LOG_ERROR("Shader activation failed for GlWarpingProgram.");
        }

        if (m_renderMode != 0) {
            if (!m_renderMode->Activate()) {
                CANDERA_DEVICE_LOG_WARN("Render mode activation failed for GlWarpingProgram.");
            }
        }

        static_cast<void>(RenderDevice::ActivateRenderState(RenderDevice::DitheringEnable, 0));
        static_cast<void>(RenderDevice::ActivateRenderState(RenderDevice::SampleAlphaToCoverageMaskEnable, 0));
        static_cast<void>(RenderDevice::ActivateRenderState(RenderDevice::SampleCoverageMaskEnable, 0));
        static_cast<void>(RenderDevice::ActivateRenderState(RenderDevice::ScissorTestEnable, 0));

        if (m_textureUniformLocation != -1) {
            Int textureUnit[] = { 0 };
            if (!RenderDevice::SetUniform(m_textureUniformLocation, &textureUnit[0], Shader::Integer)) {
                CANDERA_DEVICE_LOG_ERROR("Set uniform texture failed.");
            }
        }

        UpdateInvertUniform();
    }
}

void GlWarpingProgram::UpdateInvertUniform() const
{
    if (m_invertUniformLocation == -1) {
        return;
    }

    DisplayOrientation::Enum orientation = GetDisplayOrientation();

    GLfloat invertUniform[4] = { 1.0F, -1.0F, 0.0F, 1.0F };

    if ((static_cast<UInt>(orientation) & static_cast<UInt>(DisplayOrientation::HorizontallyFlipped)) != 0) {
        invertUniform[0] = -1.0F;
        invertUniform[2] = 1.0F;
    }

    if ((static_cast<UInt>(orientation) & static_cast<UInt>(DisplayOrientation::VerticallyFlipped)) != 0) {
        invertUniform[1] = 1.0F;
        invertUniform[3] = 0.0F;
    }

    if (!RenderDevice::SetUniform(m_invertUniformLocation, &invertUniform[0], Shader::FloatVec4)) {
        CANDERA_DEVICE_LOG_ERROR("Set uniform invert failed.");
    }
}

void GlWarpingProgram::UpdateWarpWeightUniform(const Float* warpWeight, Int warpWeightCount) const
{
    if (m_warpWeightUniformLocation == -1) {
        return;
    }

    if (!RenderDevice::SetUniform(m_warpWeightUniformLocation, warpWeight, Shader::Float, warpWeightCount)) {
        CANDERA_DEVICE_LOG_ERROR("Set uniform warp weight failed.");
    }
}
}
