//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(GL_WARPING_MANAGER_H)
#define GL_WARPING_MANAGER_H

#include <CanderaPlatform/Device/Common/Internal/GL/GlWarpingMesh.h>
#include <CanderaPlatform/Device/Common/Internal/GL/GlWarpingProgram.h>

namespace Candera
{

    class Display;
    class WarpingProperties;

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GlWarpingManager is a class that handles the lifetime of a 
 * GlWarpingMesh and its relationship to a ContextResourcePool.
 */
    class GlWarpingManager
    {
        public:
            /**
             * Constructor.
             */
            GlWarpingManager();

            /**
             * Destructor.
             * When this is called all resources shall already be released, as
             * the release is context dependent.
             */
            ~GlWarpingManager();

            /**
             * Attach a display and warping properties to this object.
             * @param display The display that is warped by this object.
             * @param warpingProperties The object that provides access to
             *                          warping properties.
             */
            void Initialize(
                Display& display,
                WarpingProperties& warpingProperties);

            /**
             * Create a handle that will be used to access the warping 
             * resources. This method shall be called while a context attached
             * to the context resource pool of the current display is active.
             * This is used to increase the count of the contexts that use the
             * warping resources.
             * @return A non-zero handle used for retrieving resources or
             *         0 if the current context is invalid.
             */
            Handle CreateWarpingResources();

            /**
             * Retrieve a warping mesh for this context.
             * This method shall be called while the context used for creating
             * the handle is active. The mesh is no longer valid if the context
             * changes, if warping properties change or if the handle has been
             * released.
             * @param handle A handle created by CreateWarpingResources.
             * @return Pointer to an uploaded GlWarpingMesh that can be used for
             *         drawing or 0 if the handle or the context are invalid.
             */
            const GlWarpingMesh* GetWarpingMesh(Handle handle);

            /**
             * Retrieve a warping program for this context.
             * The warping program has the same visibility as the warping mesh.
             * See GetWarpingMesh.
             * @param handle A handle created by CreateWarpingResources.
             * @return Pointer to an uploaded GlWarpingMesh that can be used for
             *         drawing or 0 if the handle or the context are invalid.
             */
            const GlWarpingProgram* GetWarpingProgram(Handle handle);

            /**
             * Release the warping resources.
             * The handle becomes invalid and shall not be used any farther.
             * This method shall be called while the context used for creating
             * the handle is active. The number of calls to this function shall
             * match the number of calls to CreateWarpingResources.
             * @param handle A handle created by CreateWarpingResources.
             */
            void ReleaseWarpingResources(Handle handle);

            /**
            *  Retrieves the warp weights for the warping mesh.
            *  @return The warp weights.
            */
            const Float* GetWarpMatrixWeight() const;

            /**
            *  Retrieves the number of warp matrices set
            *  @return Total number of set warp matrices.
            */
            Int GetWarpMatrixCount() const;

        private:

            bool UploadWarpingResources(Handle handle);
            bool UploadWarpingProgram();
            bool UploadWarpingMesh();

            enum  { c_maxWarpMatrixCount = 8 };

            Display* m_display;
            WarpingProperties* m_warpingProperties;

            UInt m_resourceReferenceCount;

            GlWarpingMesh m_warpingMesh;
            GlWarpingProgram m_warpingProgram;

            Float m_warpWeight[c_maxWarpMatrixCount];
    };

/** @}*/ //end of CommonDevice

}

#endif
