//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlWarpingManager.h"

#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <CanderaPlatform/Device/Common/EGL/EglTraceMapper.h>

#include <CanderaPlatform/Device/Common/Base/Display.h>
#include <CanderaPlatform/Device/Common/EGL/EglWrapper.h>

namespace Candera
{
    using namespace Internal;
GlWarpingManager::GlWarpingManager() :
    m_display(0),
    m_warpingProperties(0),
    m_resourceReferenceCount(0),
    m_warpingMesh(),
    m_warpingProgram()
{
    m_warpWeight[0] = 1.0F;
    for (UInt i = 1; i < c_maxWarpMatrixCount; ++i) {
        m_warpWeight[i] = 0.0F;
    }
}

GlWarpingManager::~GlWarpingManager()
{
    if (m_resourceReferenceCount > 0) {
        CANDERA_DEVICE_LOG_WARN("There are resource that were not released");
    }
}

void GlWarpingManager::Initialize(
    Display& display,
    WarpingProperties& warpingProperties)
{
    if ((m_resourceReferenceCount > 0) || (m_display != 0)) {
        CANDERA_DEVICE_LOG_ERROR("Already initialized.");
        return;
    }

    m_display = &display;
    m_warpingProperties = &warpingProperties;
}

Handle GlWarpingManager::CreateWarpingResources()
{
    if (m_display == 0) {
        CANDERA_DEVICE_LOG_ERROR("Not initialized.");
        return 0;
    }
    EGLContext context = EGLWrapper::GetInstance().GetCurrentContext();
    if (context == EGL_NO_CONTEXT){
        CANDERA_DEVICE_LOG_ERROR("Current Egl context is invalid for warping.");
        return 0;
    }
    m_resourceReferenceCount ++;
    return FeatStd::Internal::PointerToScalar<Handle>(context);
}

const GlWarpingMesh* GlWarpingManager::GetWarpingMesh(Handle handle)
{
    if (UploadWarpingResources(handle)) {
        return &m_warpingMesh;
    }
    else {
        return 0;
    }
}

const GlWarpingProgram* GlWarpingManager::GetWarpingProgram(Handle handle)
{
    if (UploadWarpingResources(handle)) {
        return &m_warpingProgram;
    }
    else {
        return 0;
    }
}

const Float* GlWarpingManager::GetWarpMatrixWeight() const
{
    if (0 == m_warpingProperties) {
        CANDERA_DEVICE_LOG_ERROR("Not initialized.");
        return &m_warpWeight[0];
    }

    return m_warpingProperties->GetActualWarpMatrixWeight();
}

Int GlWarpingManager::GetWarpMatrixCount() const 
{
    if (0 != m_warpingProperties) {
        return m_warpingProperties->GetActualWarpMatrixCount();
    }
    return c_maxWarpMatrixCount;
}

void GlWarpingManager::ReleaseWarpingResources(Handle handle)
{
    if (m_display == 0) {
        CANDERA_DEVICE_LOG_ERROR("Not initialized.");
        return;
    }
    EGLContext context = EGLWrapper::GetInstance().GetCurrentContext();
    if ((handle == 0) || (FeatStd::Internal::PointerToScalar<Handle>(context) != handle)) {
        CANDERA_DEVICE_LOG_ERROR("The warping mesh handle is "
            "not available in the current context.");
        return;
    }
    if (m_resourceReferenceCount == 0){
        CANDERA_DEVICE_LOG_ERROR("The warping mesh handle is invalid.");
        return;
    }

    m_resourceReferenceCount--;
    if (m_resourceReferenceCount == 0){
        m_warpingMesh.Unload();
        m_warpingProgram.Unload();
    }

    m_warpingProperties->SetWarpMatrixDirtyFlag(true);
}

bool GlWarpingManager::UploadWarpingResources(Handle handle)
{
    if (m_display == 0) {
        CANDERA_DEVICE_LOG_ERROR("Not initialized.");
        return false;
    }

    EGLContext context = EGLWrapper::GetInstance().GetCurrentContext();
    if ((handle == 0) || (FeatStd::Internal::PointerToScalar<Handle>(context) != handle)) {
        CANDERA_DEVICE_LOG_ERROR("The resource handle is "
            "not available in the current context.");
        return false;
    }

    if (m_resourceReferenceCount == 0) {
        CANDERA_DEVICE_LOG_ERROR("The resource handle is invalid.");
        return false;
    }
    
    bool rc = UploadWarpingMesh();
    rc = rc && UploadWarpingProgram();

    return rc;
}

bool GlWarpingManager::UploadWarpingMesh()
{
    if (!m_warpingProperties->IsWarpMatrixDirty()) {
        return true;
    }

    m_warpingProgram.Unload();
    m_warpingMesh.Unload();
    m_warpingProperties->SetWarpMatrixDirtyFlag(false);

    if (!m_warpingMesh.IsUploaded()) {
        bool warpingEnabled = m_warpingProperties->IsWarpingEnabled() &&
            (m_warpingProperties->IsWarpingMatrixValid());
        if (warpingEnabled) {
            bool rc = m_warpingMesh.Upload(m_warpingProperties->m_warpMatrix, m_warpingProperties->c_maxWarpMatrixCount, m_warpingProperties->m_warpImageBounds);
            if (!rc) {
                CANDERA_DEVICE_LOG_ERROR("The display warping mesh upload failed.");
                return false;
            }
        }
        else {
            const Float c_data[] = {
                0.0F, 0.0F,
                1.0F, 0.0F,
                0.0F, 1.0F,
                1.0F, 1.0F
            };
            // Fall back to default matrix.
            WarpMatrix matrix(2, 2, c_data);
            Rectangle warpImageBounds(0.0F, 0.0F, 1.0F, 1.0F);
            if (!m_warpingMesh.Upload(&matrix, 1, warpImageBounds)) {
                CANDERA_DEVICE_LOG_ERROR("The default warping mesh upload failed.");
                return false;
            }
        }
    }

    if (!m_warpingMesh.IsUploaded()) {
        CANDERA_DEVICE_LOG_ERROR("Warping mesh is not uploaded.");
        return false;
    }

    return true;
}

bool GlWarpingManager::UploadWarpingProgram()
{
    if (!m_warpingProgram.IsUploaded()) {
        Int matrixCount = m_warpingProperties->GetActualWarpMatrixCount();
        Int attributeCount = (matrixCount > 0) ? matrixCount : 1;
        if (!m_warpingProperties->IsWarpingEnabled()) {
            attributeCount = 1;
        }
        if (!m_warpingProgram.Upload(attributeCount)) {
            CANDERA_DEVICE_LOG_ERROR("The warping program upload failed.");
            return false;
        }
    }

    DisplayOrientation::Enum orientation = m_warpingProperties->GetDisplayOrientation();
    m_warpingProgram.SetDisplayOrientation(orientation);
    return true;
}

}
