//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlSimpleWarpingMesh.h"

#include <Candera/System/Mathematics/Rectangle.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <CanderaPlatform/Device/Common/OpenGLES/GlTrace.h>

namespace Candera
{
struct GlWarpingMeshVertex {
    Float x;
    Float y;
    Float tx;
    Float ty;
};

GlSimpleWarpingMesh::GlSimpleWarpingMesh() :
    m_vertexBuffer(0),
    m_drawCount(0)
{
}

GlSimpleWarpingMesh::~GlSimpleWarpingMesh()
{
    Unload();
}

bool GlSimpleWarpingMesh::Upload(WarpMatrix& src, const Rectangle& imageBounds)
{
    if (m_vertexBuffer != 0) {
        CANDERA_DEVICE_LOG_WARN("already uploaded");
        return false;
    }
    const Vector2* vertexData = src.GetVertexData();
    if (vertexData == 0) {
        CANDERA_DEVICE_LOG_WARN("warp matrix empty");
        return false;
    }
    Int width = src.GetWidth();
    Int height = src.GetHeight();

    m_drawCount = static_cast<GLsizei>(2 * width * (height - 1));

    GlWarpingMeshVertex *data = FEATSTD_NEW_ARRAY(GlWarpingMeshVertex, m_drawCount);
    if (data == 0) {
        CANDERA_DEVICE_LOG_WARN("failed allocation");
        return false;
    }

    Float xTextureStepWidth = imageBounds.GetWidth() / static_cast<Float>(width - 1);
    Float yTextureStepWidth = imageBounds.GetHeight() / static_cast<Float>(height - 1);
    GlWarpingMeshVertex* vertex = data;
    for (Int y = 0; y < (height - 1); y ++) {
        for(Int i = 0; i < width; i ++) {
            Int x = ((static_cast<UInt>(y) & 1U) != 0) ? ((width - 1) - i) : i;
            vertex->x = src.GetVertex(x, y).GetX();
            vertex->y = src.GetVertex(x, y).GetY();
            vertex->tx = imageBounds.GetPosition().GetX() + static_cast<Float>(x)* xTextureStepWidth;
            vertex->ty = imageBounds.GetPosition().GetY() + static_cast<Float>(y)* yTextureStepWidth;
            ++vertex;
            vertex->x = src.GetVertex(x, y + 1).GetX();
            vertex->y = src.GetVertex(x, y + 1).GetY();
            vertex->tx = imageBounds.GetPosition().GetX() + static_cast<Float>(x)* xTextureStepWidth;
            vertex->ty = imageBounds.GetPosition().GetY() + static_cast<Float>(y + 1) * yTextureStepWidth;
            ++vertex;
        }
    }
    glGenBuffers(1, &m_vertexBuffer);
    CANDERA_DEVICE_CHECK_AND_LOG_WARN(Gles, "glGenBuffers");

    glBindBuffer(GL_ARRAY_BUFFER, m_vertexBuffer);
    CANDERA_DEVICE_CHECK_AND_LOG_WARN(Gles, "glBindBuffer");

    UInt32 dataSize = static_cast<UInt32>(m_drawCount) *
        static_cast<UInt32>(sizeof(GlWarpingMeshVertex));
    glBufferData(GL_ARRAY_BUFFER, dataSize, data, GL_STATIC_DRAW);
    CANDERA_DEVICE_CHECK_AND_LOG_WARN(Gles, "glBufferData");

    FEATSTD_DELETE_ARRAY(data);

    return true;
}

void GlSimpleWarpingMesh::Unload()
{
    if (m_vertexBuffer != 0) {
        glDeleteBuffers(1, &m_vertexBuffer);
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Gles, "glDeleteBuffers");

        m_vertexBuffer = 0;
        m_drawCount = 0;
    }
}

void GlSimpleWarpingMesh::Activate() const
{
    if (m_vertexBuffer != 0) {
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

        // Activate vertex buffer.
        glBindBuffer(GL_ARRAY_BUFFER, m_vertexBuffer);
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Gles, "glBindBuffer");

        // Set position.
        glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 16, 0);
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Gles, "glVertexAttribPointer");

        // Set texture coordinates.
        glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, 16, FeatStd::Internal::ScalarToPointer<const void*>(Handle(8)));
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Gles, "glVertexAttribPointer");

#ifdef CANDERA_DEVICE_AMBER
        // Amber uses ES1.1 RenderDevice, however glEnableVertexAttribArray is ES2.0 and above.
        // Due to mixing GL versions in this configuration, the RenderDevice::ActivateAttribute function (that
        // also caches the vertex attribute array state) cannot be used.

        // Enable position.
        glEnableVertexAttribArray(0);
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Gles, "glEnableVertexAttribArray");

        // Enable texture coordinates.
        glEnableVertexAttribArray(1);
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Gles, "glEnableVertexAttribArray");

        RenderDevice::ResyncCurrentRenderStateCache(RenderDevice::ResyncVertexAttribArrayOption);
#else
        // Enable position.
        static_cast<void>(RenderDevice::ActivateAttribute(0));

        // Enable texture coordinates.
        static_cast<void>(RenderDevice::ActivateAttribute(1));
#endif
    }
}

void GlSimpleWarpingMesh::Draw() const
{
    if (m_drawCount != 0) {
        glDrawArrays(GL_TRIANGLE_STRIP, 0, m_drawCount);
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Gles, "glDrawArrays");
    }
}
}
