//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GL_SIMPLE_TEXTURE_FRAME_BUFFER_H)
#define GL_SIMPLE_TEXTURE_FRAME_BUFFER_H

#include <CanderaPlatform/Device/Common/OpenGLES/GlInclude.h>

#include <CanderaPlatform/Device/Common/Internal/GL/GlFrameBufferTypes.h>
#include <CanderaPlatform/Device/Common/Internal/GL/GlFrameBufferProperties.h>

#include <CanderaPlatform/Device/Common/Internal/GDU/GduSupport.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

class EglContext;

/**
 * @brief GlSimpleTextureFrameBuffer represents wrapper for a texture.
 * No actual frame buffer is generated.
 */
class GlSimpleTextureFrameBuffer
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GlSimpleTextureFrameBuffer::Unload)

    public:
        static const Int c_maxBufferCount = 1;

        /**
         *  Constructs a GlSimpleTextureFrameBuffer object.
         */
        GlSimpleTextureFrameBuffer();

        /**
         *  Destructs a GlSimpleTextureFrameBuffer object.
         */
        ~GlSimpleTextureFrameBuffer();

        /**
         *  Does nothing. Required by GduBaseRenderTarget.
         *  @param contextProvider Unused.
         *  @return True.
         */
        bool Activate(EglContext& /*contextProvider*/) { return true; }

        /**
         *  Returns current height of the texture. Required by GduBaseRenderTarget.
         *  @return The current height of the window surface.
         */
        Int GetHeight() const;

        /**
         *  Returns current width of the texture. Required by GduBaseRenderTarget.
         *  @return The current width of the window surface.
         */
        Int GetWidth() const;

        /**
         *  Uploads this object to video memory. Required by GduBaseRenderTarget.
         *  @param contextProvider  The associated context provider.
         *  @param support          Unused.
         *  @param properties       Properties to which this object should be set.
         *  @return True if successful, false otherwise.
         */
        bool Upload(EglContext& contextProvider, const GduSupport& support, const GlFrameBufferProperties& properties);

        /**
         *  Unloads this object from video memory. Required by GduBaseRenderTarget.
         *  @param contextProvider Object providing the Egl context in which this object resides.
         */
        void Unload(EglContext& contextProvider);

        /**
         *  Does nothing. Required by GduBaseRenderTarget.
         *  @param contextProvider  Unused.
         *  @param properties       Unused.
         */
        void ApplyChanges(EglContext& contextProvider, const GlFrameBufferProperties& properties) const;

        /**
         *  Does nothing. Required by GduBaseRenderTarget.
         *  @param contextProvider  Unused.
         */
        void SwapBuffers(EglContext& contextProvider) const;

        /**
         *  Returns the image handle associated with the given index. Required by GenericImageSource.
         *  @param bufferId         Must be 0.
         *  @param imageId          Must be 0.
         *  @return Image handle associated with this object.
         */
        Handle GetImageHandle(Int bufferId, Int imageId) const;

        /**
         *  Returns the buffer that should be used as image source.
         *  @return 0.
         */
        Int GetPreviousBufferIndex() const { return 0; }

        /**
         *  Allow use as 2D frame buffer.
         *  @return Same as GetHeight.
         */
        Int GetHeight2D() const { return GetHeight(); }

        /**
         *  Allow use as 2D frame buffer.
         *  @return Same as GetWidth.
         */
        Int GetWidth2D() const { return GetWidth(); }

        /**
         *  Does nothing.
         *  @param contextProvider Unused.
         *  @return True.
         */
        bool Activate2D(EglContext& /*contextProvider*/) { return true; }

        /**
         *  Initialize drawing to this frame buffer.
         *  @param contextProvider Object providing the Egl context in which this frame buffer resides.
         */
        void BeginDraw(EglContext& /*contextProvider*/) {}

        /**
         *  Ends drawing to this frame buffer.
         *  @param contextProvider Unused.
         */
        void EndDraw(EglContext& /*contextProvider*/) {}

        /**
         *  Initialize 2D drawing to this frame buffer.
         *  @param contextProvider Object providing the Egl context in which this frame buffer resides.
         */
        void BeginDraw2D(EglContext& /*contextProvider*/) {}

        /**
         *  Ends 2D drawing to this frame buffer.
         *  @param contextProvider Unused.
         */
        void EndDraw2D(EglContext& /*contextProvider*/) {}

    private:
        GLuint m_colorTexture;
        GlFrameBufferProperties m_properties;
};

/** @}*/ //end of CommonDevice

}

#endif
