//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GL_DEFAULT_FRAME_BUFFER_H)
#define GL_DEFAULT_FRAME_BUFFER_H


#include <CanderaPlatform/Device/Common/OpenGLES/GlInclude.h>
#include <CanderaPlatform/Device/Common/OpenGLES/GlTrace.h>
#include <CanderaPlatform/Device/Common/EGL/EglTraceMapper.h>

#include <CanderaPlatform/Device/Common/Internal/EGL/EglChain.h>
#include <CanderaPlatform/Device/Common/Internal/GDU/GduProperties.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GlDefaultFrameBuffer represents the default Open GL frame buffer object. Activating this
 * frame buffer will bind the frame buffer named 0.
 *
 * This object may be used as a frame buffer in a GduBaseRenderTarget, with a 3D attachment.
 */
template <typename TSupport>
class GlDefaultFrameBuffer
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GlDefaultFrameBuffer<TSupport>::Unload)

    public:
        typedef TSupport Support;
        typedef GduProperties Properties;

        static const Int c_maxBufferCount = 0;

        /**
         *  Constructs a GlDefaultFrameBuffer object.
         */
        GlDefaultFrameBuffer();

        /**
         *  Destructs a GlDefaultFrameBuffer object.
         */
        ~GlDefaultFrameBuffer();

        /**
         *  Activates the frame buffer object. Required by GduBaseRenderTarget.
         *  @param contextProvider  Unused.
         *  @return True, if no errors occurred, false otherwise.
         */
        bool Activate(EglChain& contextProvider);

        /**
         *  Returns current height of the surface. Required by GduBaseRenderTarget.
         *  @return The current height of the window surface.
         */
        Int GetHeight() const;

        /**
         *  Returns current width of the surface. Required by GduBaseRenderTarget.
         *  @return The current width of the window surface.
         */
        Int GetWidth() const;

        /**
         *  Uploads this object to video memory. Required by GduBaseRenderTarget.
         *  @param contextProvider  Unused.
         *  @param support          Parameter that suplies the frame buffer dimentions.
         *  @param properties       Unused.
         *  @return True.
         */
        bool Upload(EglChain& contextProvider, Support& support, const Properties& properties);

        /**
         *  Resets the state of this object to uninitilized. Required by GduBaseRenderTarget.
         *  @param contextProvider  Unused.
         */
        void Unload(EglChain& contextProvider);

        /**
         *  Does nothing. Required by GduBaseRenderTarget.
         *  @param contextProvider  Unused.
         *  @param properties       Unused.
         */
        void ApplyChanges(EglChain& contextProvider, const Properties& properties);

        /**
         *  Swaps the buffers of the associated EGLSurface.
         *  @param contextProvider  parameter which provides the surface and the display.
         */
        void SwapBuffers(EglChain& contextProvider);

       /**
         *  Returns the support object attached to this frame buffer.
         *  @return 0.
         */
        Support* GetSupport() { return m_support; }

        Int GetHeight2D() const { return GetHeight(); }
        Int GetWidth2D() const { return GetWidth(); }
        bool Activate2D(EglChain& contextProvider) { return Activate(contextProvider); }

        void BeginDraw(EglChain& contextProvider) { Activate(contextProvider); }
        void EndDraw(EglChain& /*contextProvider*/) {}

        void BeginDraw2D(EglChain& contextProvider) { Activate2D(contextProvider); }
        void EndDraw2D(EglChain& /*contextProvider*/) {}

    private:
        Support *m_support;
};

/** @}*/ //end of CommonDevice


template <typename TSupport>
GlDefaultFrameBuffer<TSupport>::GlDefaultFrameBuffer() :
    m_support(0)
{
}

template <typename TSupport>
GlDefaultFrameBuffer<TSupport>::~GlDefaultFrameBuffer()
{
}

template <typename TSupport>
bool GlDefaultFrameBuffer<TSupport>::Activate(EglChain& /*contextProvider*/)
{
    if (m_support != 0) {
        glBindFramebuffer(GL_FRAMEBUFFER, 0);
        CANDERA_DEVICE_CHECK_AND_LOG_DEBUG(Gles, "glBindFrameBuffer");
        return !CANDERA_DEVICE_PACKAGE_TRACE_CLASS(Gles)::WasLastErrorSet();
    }
    return false;
}

template <typename TSupport>
Int GlDefaultFrameBuffer<TSupport>::GetHeight() const
{
    return (m_support == 0) ? 0 : m_support->GetHeight();
}

template <typename TSupport>
Int GlDefaultFrameBuffer<TSupport>::GetWidth() const
{
    return (m_support == 0) ? 0 : m_support->GetWidth();
}

template <typename TSupport>
bool GlDefaultFrameBuffer<TSupport>::Upload(EglChain& /*contextProvider*/, Support& support, const Properties& /*properties*/)
{
    m_support = &support;
    return true;
}

template <typename TSupport>
void GlDefaultFrameBuffer<TSupport>::ApplyChanges(EglChain& /*contextProvider*/, const Properties& /*properties*/)
{
}

template <typename TSupport>
void GlDefaultFrameBuffer<TSupport>::SwapBuffers(EglChain& contextProvider)
{
    if (contextProvider.GetDisplay() != EGL_NO_DISPLAY) {
        eglSwapBuffers(contextProvider.GetDisplay(), contextProvider.GetDrawSurface());
        CANDERA_DEVICE_CHECK_AND_LOG_WARN(Egl, "eglSwapBuffers");
    }
}

template <typename TSupport>
void GlDefaultFrameBuffer<TSupport>::Unload(EglChain& /*contextProvider*/)
{
    m_support = 0;
}

}

#endif
