//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(GL_BINARY_SHADER_OBJECT_H)
#define GL_BINARY_SHADER_OBJECT_H


#include <FeatStd/MemoryManagement/SharedPointer.h>

#include <Candera/Engine3D/Core/Shader.h>


namespace Candera {


/** @addtogroup CommonDevice
 *  @{
 */


/**
 *  @brief Class encapsulating binary shader or binary shaders program code, its size and Shader::BuildType. Created from GlBinaryShaderObject::RawDataPointer.
 */
class GlBinaryShaderObject {
public:
    // Types

    typedef FeatStd::MemoryManagement::SharedPointer<GlBinaryShaderObject> SharedPointer;

    /**
     * Element of raw binary data with preamble, followed by size and actual binary shader or binary shaders program code.
     */
    typedef void RawData;

    /**
     * Element of binary shader or binary shaders program code.
     */
    typedef void BinaryCode;

    /**
     * Size of encapsulated binary shader or binary shaders program code.
     */
    typedef Candera::UInt32 BinaryCodeSize;

    /**
    * Raw data preamble. Contains Shader::BuildType.
    */
    typedef Candera::UInt32 Preamble;


    // Constants

    /**
     * Length of header with preamble and size data, in bytes.
     */
    enum { c_headerLengthBytes = sizeof(Preamble) + sizeof(BinaryCodeSize) };


public:
    // Creation and verification

    /**
     * Parse raw data and return new GlBinaryShaderObject if the data is valid, or NULL SharedPointer otherwise.
     * Note that the lifetime of rawData must at least match the lifetime of created GlBinaryShaderObject instance, otherwise any subsequent calls to its GetBinaryCode() will return dangling pointer.
     * @param rawData Pointer to raw (i.e. not parsed yet) binary data with preamble, followed by size and actual binary shader or binary shaders program code.
     * @return SharedPointer to new GlBinaryShaderObject encapsulating parsed binary data.
     */
    static SharedPointer Create(const RawData* rawData);

    /**
     * Parse build type from preamble of provided data.
     * @param rawData Pointer to raw (i.e. not parsed yet) binary data with preamble, followed by size and actual binary shader or binary shaders program code.
     * @param buildType Output parameter that will be updated from data.
     * @return True if the data start with valid preamble, false otherwise.
     */
    static bool GetBuildType(const RawData* rawData, Shader::BuildType& buildType);

    /**
     * Parse preamble of provided data, check its validity including build type. Parse and check size of binary shader or binary shaders program data.
     * @param rawData Pointer to raw (i.e. not parsed yet) binary data with preamble, followed by size and actual binary shader or binary shaders program code.
     * @return True if preamble if valid, false otherwise.
     */
    static bool IsPreambleValid(const RawData* rawData);

    /**
     * Write header consisting of preamble according to provided buildType, followed by size.
     * Method expects rawData to contain enough preallocated space to hold at least c_headerLengthBytes.
     * @param rawData Pointer to raw data which first bytes will be overwritten by preamble and size.
     * @param buildType Shader build type.
     * @param size Binary code size.
     */
    static void WriteHeader(RawData* rawData, Shader::BuildType buildType, BinaryCodeSize size);

    // Accessors

    /**
     * Get pointer to binary shader or binary shaders program code.
     * @return Pointer to parsed binary data after preamble.
     */
    const BinaryCode* GetBinaryCode() const { return m_binaryCode; }

    /**
     * Get binary shader or binary shaders program code size.
     * @return Size of parsed binary data without preamble.
     */
    BinaryCodeSize GeBinaryCodeSize() const { return m_size; }

    /**
     * Get build type of encapsulated binary shader or binary shaders program code.
     * @return Shader::BuildType::ShaderBinary or Shader::BuildType::ShaderProgramBinary.
     */
    Shader::BuildType GetBuildType() const { return m_buildType; }


private:
    FEATSTD_SHARED_POINTER_DECLARATION();


private:
    /**
     * Private constructor - use Create() to create new instance from RawDataPointer.
     * @param buildType Type of encapsulated binary shader code, must be one of Shader::BuildType::ShaderBinary or Shader::BuildType::ShaderProgramBinary.
     * @param size Binary shader or binary shaders program code size.
     * @param binaryCode Pointer to binary shader or binary shaders program code.
     */
    GlBinaryShaderObject(Shader::BuildType buildType, BinaryCodeSize size, const BinaryCode* binaryCode);

    // Parsing functions

    static bool ParseBuildType(Preamble preamble, Shader::BuildType& buildType);
    static Preamble GetPreamble(const RawData* rawData);
    static bool IsPreambleValid(Preamble preamble);

private:
    static const Preamble c_preamble;
    static const Preamble c_preambleMask;
    static const Preamble c_buildTypeMask;

    Shader::BuildType m_buildType;
    BinaryCodeSize m_size;
    const BinaryCode* m_binaryCode;
};


/** @}*/ // end of CommonDevice


} // namespace Candera


#endif // GL_BINARY_SHADER_OBJECT_H
