//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#include <Candera/Candera.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageTrace.h>
#include <Candera/System/MemoryManagement/Endianness.h>

#include "GlBinaryShaderObject.h"


using namespace FeatStd;
using namespace FeatStd::Internal;


namespace Candera {

FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaPlatformDevice);


// Constants


const GlBinaryShaderObject::Preamble GlBinaryShaderObject::c_preamble = 0xF0F0F000;
const GlBinaryShaderObject::Preamble GlBinaryShaderObject::c_preambleMask = 0xFFFFFF00;
const GlBinaryShaderObject::Preamble GlBinaryShaderObject::c_buildTypeMask = ~c_preambleMask;


// Helper functions

namespace ShaderObjectHelper {
    template<typename Type>
    static Type GetAtIndex(SizeType index, const GlBinaryShaderObject::RawData* rawData)
    {
        const Type value = PointerToPointer<const UInt32*>(rawData)[index];
        // rawData are in x86 native endianness
        return Candera::MemoryManagement::LittleToHostEndian(value);
    }

    template<typename Type>
    static void SetAtIndex(SizeType index, GlBinaryShaderObject::RawData* rawData, Type value)
    {
        // rawData are in host endianness
        value = Candera::MemoryManagement::HostToLittleEndian(value);
        PointerToPointer<Type*>(rawData)[index] = value;
    }
} // ShaderObjectHelper

// Interface


GlBinaryShaderObject::GlBinaryShaderObject(Shader::BuildType buildType, BinaryCodeSize size, const BinaryCode* binaryCode) :
    m_buildType(buildType),
    m_size(size),
    m_binaryCode(binaryCode)
{
    FEATSTD_DEBUG_ASSERT(m_buildType == Shader::ShaderBinary || m_buildType == Shader::ShaderProgramBinary);
    FEATSTD_DEBUG_ASSERT(m_size > 0);
    FEATSTD_DEBUG_ASSERT(m_binaryCode != 0);
}

GlBinaryShaderObject::SharedPointer GlBinaryShaderObject::Create(const RawData* rawData)
{
    GlBinaryShaderObject::SharedPointer retValue(0);

    if (rawData == 0) {
        CANDERA_DEVICE_LOG_ERROR("NULL pointer passed to GlBinaryShaderObject::Create().");
        return retValue;
    }

    const Preamble preamble = GetPreamble(rawData);
    if (IsPreambleValid(preamble)) {
        const BinaryCodeSize size = Candera::ShaderObjectHelper::GetAtIndex<BinaryCodeSize>(1, rawData);
        if (size != 0) {
            FEATSTD_COMPILETIME_ASSERT(sizeof(UInt8) == 1);
            const BinaryCode* binaryCode = PointerAdd(rawData, sizeof(Preamble) + sizeof(BinaryCodeSize));
            Shader::BuildType buildType;
            if (ParseBuildType(preamble, buildType)) {
                retValue = SharedPointer(FEATSTD_NEW(GlBinaryShaderObject)(buildType, size, binaryCode));
            }
        } else {
            CANDERA_DEVICE_LOG_ERROR("Zero GlBinaryShaderObject data size.");
        }
    } else {
        CANDERA_DEVICE_LOG_ERROR("Invalid GlBinaryShaderObject data preamble 0x%08x.", static_cast<UInt>(preamble));
    }
    return retValue;
}


bool GlBinaryShaderObject::GetBuildType(const RawData* rawData, Shader::BuildType &buildType)
{
    if (rawData == 0) {
        CANDERA_DEVICE_LOG_ERROR("NULL pointer passed to GlBinaryShaderObject::GetBuildType().");
        return false;
    }
    const Preamble preamble = GetPreamble(rawData);
    return IsPreambleValid(preamble) && ParseBuildType(preamble, buildType);
}

bool GlBinaryShaderObject::IsPreambleValid(const RawData* rawData)
{
   if (rawData == 0) {
        CANDERA_DEVICE_LOG_ERROR("NULL pointer passed to GlBinaryShaderObject::GetBuildType().");
        return false;
    }
    return IsPreambleValid(GetPreamble(rawData));
}


void GlBinaryShaderObject::WriteHeader(RawData* rawData, Shader::BuildType buildType, BinaryCodeSize size)
{
    const Preamble preamble = c_preamble | (static_cast<UInt32>(buildType) & c_buildTypeMask);
    FEATSTD_DEBUG_ASSERT(IsPreambleValid(preamble));
    ShaderObjectHelper::SetAtIndex(0U, rawData, preamble);
    FEATSTD_DEBUG_ASSERT(size > 0);
    ShaderObjectHelper::SetAtIndex(1U, rawData, size);
}


// Implementation


GlBinaryShaderObject::Preamble GlBinaryShaderObject::GetPreamble(const RawData* rawData)
{
   if (rawData == 0) {
        CANDERA_DEVICE_LOG_ERROR("NULL pointer passed to GlBinaryShaderObject::GetPreamble().");
        return false;
    }
   return ShaderObjectHelper::GetAtIndex<Preamble>(0, rawData);
}

bool GlBinaryShaderObject::IsPreambleValid(Preamble preamble)
{
    return (preamble & c_preambleMask) == c_preamble;
}

bool GlBinaryShaderObject::ParseBuildType(Preamble preamble, Shader::BuildType& buildType)
{
    const UInt32 rawBuildType = preamble & c_buildTypeMask;
    switch (rawBuildType) {
    case Shader::ShaderBinary:
        buildType = Shader::ShaderBinary;
        break;
    case Shader::ShaderProgramBinary:
        buildType = Shader::ShaderProgramBinary;
        break;
    default:
        CANDERA_DEVICE_LOG_ERROR("Invalid GlBinaryShaderObject raw data Shader::BuildType %u.", static_cast<UInt>(rawBuildType));
        return false;
    }
    return true;
}


} // namespace Candera
