//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_WRAPPER_GRAPHIC_DEVICE_UNIT_EXPOSER_H)
#define GDU_WRAPPER_GRAPHIC_DEVICE_UNIT_EXPOSER_H

#include <CanderaPlatform/Device/Common/Internal/GDU/GduAttachment.h>
#include <CanderaPlatform/Device/Common/Internal/GDU/GduWrapper.h>
#include <CanderaPlatform/Device/Common/Internal/GDU/GduProperties.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief Class GduWrapperExposer provides dedicated interfaces to access an object as a
 * specific gdu attachment.
 *
 * The wrapped type must implement the folowing methods:
 *  void SetGraphicDeviceUnit(GraphicDeviceUnit*);
 * The wrapped type must be a child of TExposedType.
 *
 * @param TWrappedType type of the wrapped object
 * @param TExposedType type by which the wrapped object is exposed. Will be one of
 *   ImageSource3D, ImageSource2D, RenderTarget3D and RenderTarget2D.
 */
template <typename TWrappedType, typename TExposedType>
class GduWrapperExposer : public GduWrapper<TWrappedType>
{
public:
    FEATSTD_TYPEDEF_BASE(GduWrapper<TWrappedType>);
    typedef typename Base::WrappedType WrappedType;
    typedef TExposedType ExposedType;

    GduWrapperExposer() : m_gdu(0) {}

    /**
     * Initilize the wrapper. Called when creating a GduAttachment.
     * Overrides homonymic method from GduWrapper
     * @param wrappedObject     Base object of the gdu, wrapped by this wrapper.
     * @param index             Index of the wrapper within the Gdu.
     */
    void SetBaseObject(WrappedType& wrappedObject, Int index);

    /**
     * Record the current gdu to store within the wrapped object. Called when creating a GduAttachment.
     * @param gdu     Pointer the GraphicDeviceUnit that is parent to the wrapped object.
     */
    void SetGraphicDeviceUnit(GraphicDeviceUnit* gdu);

private:
    GraphicDeviceUnit* m_gdu;
};

/**
 * @brief Specialization that exposes the wrapped object as the exposed type.
 */
template <typename TWrappedType, typename TExposedType>
class GduAttachmentConvertor<GduWrapperExposer<TWrappedType, TExposedType>, TExposedType, false>
{
public:
    typedef GduWrapperExposer<TWrappedType, TExposedType> ElementType;
    static TExposedType* Convert(GduWrapperExposer<TWrappedType, TExposedType>* el) {
        return el->GetWrappedObject();
    }
};

/** @}*/ //end of CommonDevice

template <typename TWrappedType, typename TExposedType>
void GduWrapperExposer<TWrappedType, TExposedType>::SetBaseObject(WrappedType& wrappedObject, Int index)
{
    Base::SetBaseObject(wrappedObject, index);
    Base::GetWrappedObject()->SetGraphicDeviceUnit(m_gdu);
}

template <typename TWrappedType, typename TExposedType>
void GduWrapperExposer<TWrappedType, TExposedType>::SetGraphicDeviceUnit(GraphicDeviceUnit* gdu)
{
    m_gdu = gdu;
}


}

#endif
