//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_RENDERTARGET_3D_ATTACHMENT_H)
#define GDU_RENDERTARGET_3D_ATTACHMENT_H

#include <Candera/Environment.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GduRenderTarget3DAttachment should be attached to a GduRenderTarget3DAttachment,
 * to make a complete RenderTarget3D, and interface for GduWrapperImageSource3D.
 *
 * It introduces a set of additional restrictions on the aggregated object:
 * - the Context needs to be a 3D ContextProvider. Must implement:
 *   bool Activate();
 *   void* GetContext();
 *   bool IsUploaded() const;
 * - the FrameBuffer needs to implement:
 *   Int GetHeight() const;
 *   Int GetWidth() const;
 *   bool Activate(TBase& );
 *  - SyncObject must implement:
 *   void WaitBeginDraw(TBase& );
 *   void MarkEndDraw(TBase& )
 *   void WaitSync(TBase& )
 *   void Sync(TBase& );
 *
 * @param TBase GduRenderTarget3DAttachment or class that extends GduRenderTarget3DAttachment
 */
template <typename TBase>
class GduRenderTarget3DAttachment : public TBase
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GduRenderTarget3DAttachment::Unload)

    public:
        FEATSTD_TYPEDEF_BASE(TBase);
        typedef typename Base::FrameBuffer FrameBuffer;
        typedef typename Base::Properties Properties;
        typedef typename Base::SyncObject SyncObject;
        typedef typename Base::Support Support;

        /**
         *  Constructs a GduRenderTarget3DAttachment object.
         */
        GduRenderTarget3DAttachment() {}

        /**
         *  Destructs a GduRenderTarget3DAttachment object.
         */
        ~GduRenderTarget3DAttachment() {}

        /**
         *  Retrieves the context associated with this RenderTarget.
         *  @return handle of the associated EGLContext.
         */
        void* GetContextHandle() const;

        /**
         *  Activates this render target and makes it the current default render target.
         *  @return True, if no errors occurred, false otherwise.
         */
        bool Activate();

        /**
         *  Returns current height of the surface. Implements function of class Surface.
         *  @return The current height of the surface.
         */
        Int GetHeight() const;

        /**
         *  Returns current width of the surface. Implements function of class Surface.
         *  @return The current width of the surface.
         */
        Int GetWidth() const;

        /**
         *  Implements function of class RenderTarget.
         */
        void BeginDraw();

        /**
         *  Implements function of class RenderTarget.
         */
        void EndDraw();

        /**
         *  Implements function of class Synchronizable.
         */
        void WaitSync();

        /**
         *  Implements function of class Synchronizable.
         */
        void Sync();

        /**
         *  Implements function of class Synchronizable.
         */
        void OnAccess();
};

/** @}*/ //end of CommonDevice

template <typename TBase>
void* GduRenderTarget3DAttachment<TBase>::GetContextHandle() const
{
    if (!Base::IsUploaded()) {
        return 0;
    }
    typedef typename Base::Base Context;
    return Context::GetContext();
}

template <typename TBase>
bool GduRenderTarget3DAttachment<TBase>::Activate()
{
    if (!Base::IsUploaded()) {
        return false;
    }
    if (Base::Activate()) {
        return Base::GetFrameBuffer().Activate(*this);
    }
    return false;
}

template <typename TBase>
Int GduRenderTarget3DAttachment<TBase>::GetHeight() const
{
    return Base::IsUploaded() ? Base::GetFrameBuffer().GetHeight() : 0;
}

template <typename TBase>
Int GduRenderTarget3DAttachment<TBase>::GetWidth() const
{
    return Base::IsUploaded() ? Base::GetFrameBuffer().GetWidth() : 0;
}

template <typename TBase>
void GduRenderTarget3DAttachment<TBase>::BeginDraw()
{
    if (!Base::IsUploaded()) {
        return;
    }
    if (Base::IsDrawing()){
        CANDERA_DEVICE_LOG_WARN("already drawing");
        return;
    }
    Base::SetDrawing(true);
    if (Base::Activate()) {
        Base::GetFrameBuffer().BeginDraw(*this);
    }
}

template <typename TBase>
void GduRenderTarget3DAttachment<TBase>::EndDraw()
{
    if (!Base::IsDrawing()){
        CANDERA_DEVICE_LOG_WARN("not drawing");
        return;
    }
    Base::GetFrameBuffer().EndDraw(*this);
    Base::GetSyncObject().MarkEndDraw(*this);
    Base::SetDrawing(false);
}

template <typename TBase>
void GduRenderTarget3DAttachment<TBase>::WaitSync()
{
    if (Base::IsUploaded()) {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::WaitSync, "RenderTarget3D"));
        Base::GetSyncObject().WaitSync(*this);
    }
}

template <typename TBase>
void GduRenderTarget3DAttachment<TBase>::Sync()
{
    if (Base::IsUploaded()) {
        Base::GetSyncObject().Sync(*this);
    }
}

template <typename TBase>
void GduRenderTarget3DAttachment<TBase>::OnAccess()
{
    if (Base::IsUploaded()) {
        Base::GetSyncObject().MarkEndDraw(*this);
    }
}


}

#endif
