//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_PROPERTIES_SIZE_EXTENSION_H)
#define GDU_PROPERTIES_SIZE_EXTENSION_H

#include <Candera/Environment.h>
#include <FeatStd/Util/Optional.h>

namespace Candera
{

template <typename TBaseProperties>
class GduPropertiesSizeExtension : public TBaseProperties
{
    typedef TBaseProperties BaseProperties;
public:
    GduPropertiesSizeExtension() :
        BaseProperties(), m_width(0), m_height(0) {}

    GduPropertiesSizeExtension(const BaseProperties& src) :
        BaseProperties(src), m_width(0), m_height(0) {}

    GduPropertiesSizeExtension(Int width, Int height) :
        BaseProperties(), m_width(width), m_height(height) {}

    GduPropertiesSizeExtension(const BaseProperties& src, Int width, Int height) :
        BaseProperties(src), m_width(width), m_height(height) {}

    /**
     * Set the width of the frame buffer.
     * @param width value to use as width.
     */
    void SetWidth(Int width) { m_width = width; }
    /**
     * Get the width of the frame buffer.
     * @return value set by SetWidth.
     */
    Int GetWidth() const { return m_width; }

    /**
     * Set the height of the frame buffer.
     * @param height value to use as height.
     */
    void SetHeight(Int height) { m_height = height; }
    /**
     * Get the height of the frame buffer.
     * @return value set by SetHeight.
     */
    Int GetHeight() const { return m_height; }

    /**
     * Set the width adjusted with MSAA (AntiAliasing Multi Sampling Factor)
     */
    void SetMsaaWidth(const Int msaaWidht) { m_msaaWidth = msaaWidht; }
    /**
     * Get the width adjusted with MSAA (AntiAliasing Multi Sampling Factor)
     * @return value set by SetMsaaWidth
     */
    Int GetMsaaWidth() const { return (m_msaaWidth.IsSet() ? *m_msaaWidth : m_width); }

    /**
     * Set the height adjusted with MSAA (AntiAliasing Multi Sampling Factor)
     */
    void SetMsaaHeight(const Int msaaHeight) { m_msaaHeight = msaaHeight; }
    /**
     * Get the height adjusted with MSAA (AntiAliasing Multi Sampling Factor)
     * @return value set by SetMsaaHeight
     */
    Int GetMsaaHeight() const { return (m_msaaHeight.IsSet() ? *m_msaaHeight : m_height); }

private:
    Int m_width;
    Int m_height;
    FeatStd::Optional<Int> m_msaaWidth;
    FeatStd::Optional<Int> m_msaaHeight;
};

template <
    typename TBase,
    typename TProperties,
    typename TSupport,
    typename TExtendedProperties = GduPropertiesSizeExtension<TProperties> >
class GduRenderTargetSizeExtension : public TBase
{
public:
    FEATSTD_TYPEDEF_BASE(TBase);
    typedef TProperties Properties;
    typedef TSupport Support;
    typedef TExtendedProperties ExtendedProperties;

    GduRenderTargetSizeExtension() : m_support(0) {}

    bool Upload(Int displayId, Support& support, const Properties& properties)
    {
        ExtendedProperties ext(properties);
        ext.SetWidth(support.GetWidth());
        ext.SetHeight(support.GetHeight());
        if (Base::Upload(displayId, support, ext)) {
            m_support = &support;
            return true;
        }
        return false;
    }

    void Unload()
    {
        m_support = 0;
        Base::Unload();
    }

    void ApplyChanges(const Properties& properties)
    {
        if (m_support != 0) {
            ExtendedProperties ext(properties);
            ext.SetWidth(m_support->GetWidth());
            ext.SetHeight(m_support->GetHeight());
            Base::ApplyChanges(ext);
        }
    }
private:
    Support* m_support;
};

}

#endif //GDU_PROPERTIES_SIZE_EXTENSION_H
