//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_IMAGE_2D_FRAMEBUFFER_H)
#define GDU_IMAGE_2D_FRAMEBUFFER_H

#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GduImage2DFrameBuffer is a frame buffer that wraps 3D frame buffers to obtain 2D handles.
 */
template <
    typename TBase,
    typename TContext,
    typename TSupport,
    typename TProperties
    >
class GduImage2DFrameBuffer : public TBase
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GduImage2DFrameBuffer::Unload)

    public:
        FEATSTD_TYPEDEF_BASE(TBase);
        typedef TContext Context;
        typedef TSupport Support;
        typedef TProperties Properties;

        /**
         *  Constructs a GduImage2DFrameBuffer object.
         */
        GduImage2DFrameBuffer() : m_imageCount(0) {}

        /**
         *  Destroys a GduImage2DFrameBuffer object.
         */
        ~GduImage2DFrameBuffer() {}

        /**
         *  Uploads this object to video memory. Required by GduBaseRenderTarget.
         *  @param context          Unused
         *  @param support          Unused
         *  @param properties       Unused
         *  @return True if successful, false otherwise.
         */
        bool Upload(Context& context, const Support& support, const Properties& properties);

        /**
         *  Unloads this object from video memory. Required by GduBaseRenderTarget.
         */
        void Unload(Context& context);

        Int GetHeight2D() const { return Base::GetHeight(); }
        Int GetWidth2D() const { return Base::GetWidth(); }

         /**
         *  Retrieves the associated surface.
         *  @return reference to the SurfaceProvider used internally.
         */
        SurfaceHandle GetImage2DHandle(Int index) const;

    private:
        static const Int c_maxBufferCount = Base::c_maxBufferCount;
        Int m_imageCount;
        SurfaceHandle m_surfaces[c_maxBufferCount];
};

/** @}*/ //end of CommonDevice

template <typename TBase, typename TContext,  typename TSupport,  typename TProperties>
bool GduImage2DFrameBuffer<TBase, TContext, TSupport, TProperties>::Upload(Context& context, const Support& support, const Properties& properties)
{
    if (!Base::Upload(context, support, properties)){
        return false;
    }

    m_imageCount = 0;
    while ( m_imageCount < c_maxBufferCount) {
        Handle handle = Base::GetImageHandle(m_imageCount, 0);
        if (handle == 0) {
            break;
        }
        ++m_imageCount;
    }
    
    if (!RenderDevice2D::CreateSurfaces(m_imageCount, m_surfaces)) {
        return false;
    }

    for (Int index = 0; index < m_imageCount; index++) {
        Handle handle = Base::GetImageHandle(index, 0);
        if (!RenderDevice2D::AttachNativeHandle(
            m_surfaces[index],
            Base::GetWidth(),
            Base::GetHeight(),
            properties.GetBitmapPixelFormat(),
            handle)) {
            Unload(context);
            return false;
        }
    }

    return true;
}

template <typename TBase, typename TContext,  typename TSupport,  typename TProperties>
void GduImage2DFrameBuffer<TBase, TContext, TSupport, TProperties>::Unload(Context& context)
{
    if (context.ActivateContext()) {
        if (m_imageCount != 0) {
            RenderDevice2D::DestroySurfaces(m_imageCount, m_surfaces);
        }
    }
    Base::Unload(context);

    m_imageCount = 0;
}

template <typename TBase, typename TContext,  typename TSupport,  typename TProperties>
SurfaceHandle GduImage2DFrameBuffer<TBase, TContext, TSupport, TProperties>::GetImage2DHandle(Int index) const
{
    bool imageExists = (index == 0) &&
        (m_imageCount > Base::GetPreviousBufferIndex());
    return imageExists ? m_surfaces[Base::GetPreviousBufferIndex()] : 0;
}

}

#endif
