//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_FRAME_BUFFFER_MEASUREMENT_ATTACHMENT_H)
#define GDU_FRAME_BUFFFER_MEASUREMENT_ATTACHMENT_H

#include <Candera/Environment.h>
#include <Candera/System/Diagnostics/VideoMemoryStatistic.h>
#include <FeatStd/Util/Traits.h>
#include <FeatStd/Diagnostics/Measurable.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GduFrameBufferMeasurementAttachment should be attached to a all measurable,
 * frame buffer render targets in order to get counted in VRAM memory statistics.
 *
 * Additionally to base functions Upload and Unload call the corresponding listeners 
 * of VideoMemoryStatistic. Note only measurable framebuffers attached to Base
 * are considered for VideoMemoryStatistic. Others are simply ignored.
 *
 * @param TBase GduBaseRenderTarget or class that extends GduBaseRenderTarget.
 */
template <typename TBase>
class GduFrameBufferMeasurementAttachment : public TBase
{
    public:
        typedef TBase Base;
        typedef typename Base::FrameBuffer FrameBuffer;
        typedef typename Base::Properties Properties;
        typedef typename Base::SyncObject SyncObject;
        typedef typename Base::Support Support;

        /**
         *  Constructs a GduFrameBufferMeasurementAttachment object.
         */
        GduFrameBufferMeasurementAttachment() {}

        /**
         *  Destructs a GduFrameBufferMeasurementAttachment object.
         */
        ~GduFrameBufferMeasurementAttachment() {}

        /**
         *  Uploads base object to video memory and stores measurement result in VideoMemoryStatistic.
         *  @param displayId         Display to which this render target is attached.
         *  @param support           Object used as support for this render target.
         *  @param properties        Properties to associate to the current RenderTarget.
         *  @return True if Upload was successful, false otherwise.
         */
        bool Upload(Int displayId, Support& support, Properties& properties);

        /**
         *  Unloads this object from video memory and removes measurement result from VideoMemoryStatistic.
         */
        void Unload();

};

/** @}*/ //end of CommonDevice

template <typename TBase>
bool GduFrameBufferMeasurementAttachment<TBase>::Upload(Int displayId, Support& support, Properties& properties)
{
    bool result = Base::Upload(displayId, support, properties);
    if (result) {
        if (FeatStd::Internal::IsDerivedFrom<FrameBuffer,FeatStd::Diagnostics::Measurable>::IsDerived()) {
            Candera::Diagnostics::VideoMemoryStatistic::OnFrameBufferObjectUploaded(Base::GetFrameBuffer());
        }
    }
    return result;
}

template <typename TBase>
void GduFrameBufferMeasurementAttachment<TBase>::Unload()
{
    if (FeatStd::Internal::IsDerivedFrom<FrameBuffer,FeatStd::Diagnostics::Measurable>::IsDerived()) {
        Candera::Diagnostics::VideoMemoryStatistic::OnFrameBufferObjectUnloaded(Base::GetFrameBuffer());
    }
    Base::Unload();
}

}

#endif
