//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_CONTEXT_FROM_OWNER_H)
#define GDU_CONTEXT_FROM_OWNER_H

#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaPlatform/Device/Common/Internal/Util/Select.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief The GduContextFromOwner attachment retrieves the context of an object 
 * from an owner, stored in either the support object or in the properties
 * object.
 *
 * Properties and support are forwarded to the base class after the owner was
 * retrieved.
 * The base class must derive from ContextDelegator3D. The base class needs not
 * provide Activation or Context, as they are retrieved from the owner.
 *
 * @tparam TBase The class to which it attaches.
 * @tparam TSupport The class used as support object.
 * @tparam TProperties The class used as properties.
 * @tparam TSelector Same as either TSupport or TProperties. The class that 
 *         contains the owner.
 */
template <
    typename TBase,
    typename TSupport,
    typename TProperties,
    typename TSelector
    >
class GduContextFromOwner : public TBase
{
CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::GduContextFromOwner::Unload)

public:
    typedef TBase Base;
    typedef TSupport Support;
    typedef TProperties Properties;
    typedef TSelector Selector;
    typedef typename Base::ContextHandle ContextHandle;

    /**
     *  Constructs a GduContextFromOwner object.
     */
    GduContextFromOwner() : m_mustReleasePool(false), m_owner(0) {}

    /**
     *  Destroys a GduContextFromOwner object.
     */
    ~GduContextFromOwner() {}

    /**
     *  Sets up the owner and forwards uploading to the base class.
     *  Required by GduBase.
     *  The owner provided to this function must either be 0, or compatible
     *  with the ContextResourcePool of the base RenderTarget3D.
     *  @param displayId        Forwarded to Base.
     *  @param support          Forwarded to Base. Possible owner provider.
     *  @param properties       Forwarded to Base. Possible owner provider.
     *  @return True if base upload successful and owner valid, false 
     *  otherwise.
     */
    bool Upload(Int displayId, Support& support, Properties& properties);

    /**
     *  Releases relationship to owner and forwards unload to base class.
     *  Required by GduBase.
     */
    void Unload();

    // Override function from RenderTarget3D.
    virtual bool Activate() override;
    // Override function from GraphicsContent.
    virtual bool ActivateContext() override;

    /**
     *  Retrieve the context associated with this object.
     *  Required by GduRenderTarget3DAttachment.
     */
    ContextHandle GetContext() const;

private:
    bool m_mustReleasePool;
    RenderTarget3D* m_owner;
};


/** @}*/ //end of CommonDevice

template <typename TBase, typename TSupport,  typename TProperties, typename TSelector>
bool GduContextFromOwner<TBase, TSupport, TProperties, TSelector>::Upload(
    Int displayId,
    Support& support,
    Properties& properties)
{
    RenderTarget3D* owner = Util::Select<Support, Properties, Selector>::
        Get(support, properties).GetOwner();

    ContextResourcePool *ownerPool = 0;
    if (owner != 0) {
        if (owner != 0) {
            ownerPool = owner->GetContextResourcePool();
        }
        if (ownerPool == 0) {
            CANDERA_DEVICE_LOG_ERROR("Owner pool missing.");
            return false;
        }
        ContextResourcePool *pool = this->GetContextResourcePool();
        if (pool == 0) {
            m_mustReleasePool = true;
            this->SetContextResourcePool(ownerPool);
        }
        else {
            if (pool != ownerPool) {
                CANDERA_DEVICE_LOG_ERROR("Owner not compatible to Context resource pool.");
                return false;
            }
        }
        m_owner = owner;
    }
    return Base::Upload(displayId, support, properties);
}

template <typename TBase, typename TSupport,  typename TProperties, typename TSelector>
void GduContextFromOwner<TBase, TSupport, TProperties, TSelector>::Unload()
{
    Base::Unload();
    if (m_mustReleasePool) {
        this->SetContextResourcePool(0);
        m_mustReleasePool = false;
    }
    m_owner = 0;
}

template <typename TBase, typename TSupport,  typename TProperties, typename TSelector>
bool GduContextFromOwner<TBase, TSupport, TProperties, TSelector>::Activate()
{
    if (m_owner != 0) {
        return m_owner->Activate();
    }
    else {
        return Base::Activate();
    }
}
template <typename TBase, typename TSupport,  typename TProperties, typename TSelector>
typename GduContextFromOwner<TBase, TSupport, TProperties, TSelector>::ContextHandle 
    GduContextFromOwner<TBase, TSupport, TProperties, TSelector>::GetContext() const
{
    if (m_owner != 0) {
        return reinterpret_cast<ContextHandle*>(m_owner->GetContextHandle());
    }
    else {
        return Base::GetContext();
    }
}
template <typename TBase, typename TSupport,  typename TProperties, typename TSelector>
bool GduContextFromOwner<TBase, TSupport, TProperties, TSelector>::ActivateContext()
{
    if (m_owner != 0) {
        return m_owner->Activate();
    }
    else {
        return Base::ActivateContext();
    }
}

}

#endif
