//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(GDU_ATTACHMENT_H)
#define GDU_ATTACHMENT_H

#include <CanderaPlatform/Device/Common/Internal/GDU/GduBase.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief GduAttachmentConvertor is used by GduAttachmentSelector to convert attachments to
 * one of the four types to which a GDU is cast-able.
 *
 * Specializations of this class must define a type called ElementType,
 * that must be the same as the main specialization type, in order to use the respective selector.
 * Specializations must also implement the conversion method between the attachment type and
 * the destination cast type.
 *
 * @param TElementType type of the class that is set as attachment
 * @param TElementBase type to which the class is cast
 * @param isFirstDerivedFrom parameter that specifies whether the attachment class derives the base class.
 */
template <
    class TElementType,
    class TElementBase,
    bool isFirstDerivedFrom =
        TypeTraits::Internal::IsBaseOf<TElementBase, TElementType>::Value != 0>
class GduAttachmentConvertor
{
public:
    typedef void ElementType;
    static TElementBase* Convert(TElementType* /*el*/) {
        return 0;
    }
};

/**
 * @brief Specialization of GduAttachmentConvertor used to automaticaly cast
 * classes that extend the base class.
 */
template <class TElementType, class TElementBase>
class GduAttachmentConvertor<TElementType, TElementBase, true>
{
public:
    typedef TElementType ElementType;
    static TElementBase* Convert(TElementType* el) {
        return el;
    }
};

/**
 * @brief GduAttachmentSelector is used to select the kind of object GduAttachment will expose.
 * The selection is done using GduAttachmentConvertor.
 *
 * @param TBase GDU type to which this attachment is applied.
 * @param TItemType Type of the object that is attached.
 * @param TItemTypeValidator Should be the same as TItemType for valid attachments.
 */
template <
    class TBase,
    class TItemType,
    class TItemTypeValidator = TItemType>
class GduAttachmentSelector : public TBase
{
};

/**
 * @brief Specialization of GduAttachmentSelector that attaches a RenderTarget2D.
 */
template <
    class TBase,
    class TItemType>
class GduAttachmentSelector<
    TBase,
    TItemType,
    typename GduAttachmentConvertor<TItemType, RenderTarget2D>::ElementType> :
        public TBase
{
public:

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     * @param index index of the current attachment. If the index doesn't correspond it is delegated to another attachment.
     * @return pointer to a render target 2d object.
     */
    virtual RenderTarget2D* ToRenderTarget2D(Int index = 0)
    {
#ifdef CANDERA_2D_ENABLED
        if (index == c_Rt2DId) {
            return GduAttachmentConvertor<TItemType, RenderTarget2D>::Convert(&m_item);
        }
        else {
            return TBase::ToRenderTarget2D(index);
        }
#else
        FEATSTD_UNUSED(index);
        return 0;
#endif
    }

protected:
    static const Int c_Rt2DId = TBase::c_Rt2DId + 1;
    static const Int c_itemIndex = c_Rt2DId;

    TItemType m_item;
};

/**
 * @brief Specialization of GduAttachmentSelector that attaches a RenderTarget3D.
 */
template <
    class TBase,
    class TItemType>
class GduAttachmentSelector<
    TBase,
    TItemType,
    typename GduAttachmentConvertor<TItemType, RenderTarget3D>::ElementType> :
        public TBase
{
public:

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     * @param index index of the current attachment. If the index doesn't correspond it is delegated to another attachment.
     * @return pointer to a render target 3d object.
     */
    virtual RenderTarget3D* ToRenderTarget3D(Int index = 0)
    {
#ifdef CANDERA_3D_ENABLED
        if (index == c_Rt3DId) {
            return GduAttachmentConvertor<TItemType, RenderTarget3D>::Convert(&m_item);
        }
        else {
            return TBase::ToRenderTarget3D(index);
        }
#else
        FEATSTD_UNUSED(index);
        return 0;
#endif
    }

protected:
    static const Int c_Rt3DId = TBase::c_Rt3DId + 1;
    static const Int c_itemIndex = c_Rt3DId;

    TItemType m_item;
};

/**
 * @brief Specialization of GduAttachmentSelector that attaches an ImageSource2D.
 */
template <
    class TBase,
    class TItemType>
class GduAttachmentSelector<
    TBase,
    TItemType,
    typename GduAttachmentConvertor<TItemType, ImageSource2D>::ElementType> :
        public TBase
{
public:

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     * @param index index of the current attachment. If the index doesn't correspond it is delegated to another attachment.
     * @return pointer to a image source 2d object.
     */
    virtual ImageSource2D* ToImageSource2D(Int index = 0)
    {
#ifdef CANDERA_2D_ENABLED
        if (index == c_Is2DId) {
            return GduAttachmentConvertor<TItemType, ImageSource2D>::Convert(&m_item);
        }
        else {
            return TBase::ToImageSource2D(index);
        }
#else
        FEATSTD_UNUSED(index);
        return 0;
#endif
    }

protected:
    static const Int c_Is2DId = TBase::c_Is2DId + 1;
    static const Int c_itemIndex = c_Is2DId;

    TItemType m_item;
};

/**
 * @brief Specialization of GduAttachmentSelector that attaches an ImageSource3D.
 */
template <
    class TBase,
    class TItemType>
class GduAttachmentSelector<
    TBase,
    TItemType,
    typename GduAttachmentConvertor<TItemType, ImageSource3D>::ElementType> :
        public TBase
{
public:

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     * @param index index of the current attachment. If the index doesn't correspond it is delegated to another attachment.
     * @return pointer to a image source 3d object.
     */
    virtual ImageSource3D* ToImageSource3D(Int index = 0)
    {
#ifdef CANDERA_3D_ENABLED
        if (index == c_Is3DId) {
            return GduAttachmentConvertor<TItemType, ImageSource3D>::Convert(&m_item);
        }
        else {
            return TBase::ToImageSource3D(index);
        }
#else
        FEATSTD_UNUSED(index);
        return 0;
#endif
    }

protected:
    static const Int c_Is3DId = TBase::c_Is3DId + 1;
    static const Int c_itemIndex = c_Is3DId;

    TItemType m_item;
};

/**
 * @brief GduAttachment may be used to attach conversion object or extend the functionality of a GduBase object.
 *
 * An attached object needs to have the following methods:
 *   void SetGraphicDeviceUnit(GraphicDeviceUnit*);
 *   void SetBaseObject(GduBase::TObject&, Int index);
 *   bool Upload(Int displayId, TProperties&);
 *   void ApplyChanges(TProperties&);
 *   void Unload();
 * The attached type also needs to specify a single conversion using GduAttachmentConvertor
 * to one of the 4 types a GDU casts to: RenderTarget2D, RenderTarget3D, ImageSource2D and ImageSource3D.
 *
 * @param TBase GDU type to which this attachment is applied.
 * @param TItemType Type of the object that is attached.
 * @param TProperties properties that can be used to tweak the attachment.
 */
template <
    class TBase,
    class TItemType,
    class TProperties>
class GduAttachment : public GduAttachmentSelector<TBase, TItemType>
{
public:
    typedef TBase Base;
    typedef TItemType ItemType;

    typedef GduAttachmentSelector<TBase, TItemType> Selector;

    /**
     * Constructor
     */
    GduAttachment()
    {
        Selector::m_item.SetGraphicDeviceUnit(this);
        Selector::m_item.SetBaseObject(Base::GetObject(), Selector::c_itemIndex);
    }

    /**
     * Destructor
     */
    virtual ~GduAttachment() { Unload(); }

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     * Applies properties that can be changed after upload has been conducted.
     * Note: In order to apply changes, this function activates the RenderTarget associated.
     */
    virtual void ApplyChanges()
    {
        Base::ApplyChanges();
        Selector::m_item.ApplyChanges(Base::GetProperties());
    }

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     * @return true if upload was successful
     */
    virtual bool Upload()
    {
        if (!Base::Upload()) {
            return false;
        }

        if (!Selector::m_item.Upload(Base::GetDisplay(), Base::GetProperties())) {
            Base::Unload();
            return false;
        }

        ApplyChanges();
        return true;
    }

    /**
     * Overrides pure virtual function of class GraphicDeviceUnit.
     */
    virtual void Unload()
    {
        Selector::m_item.Unload();
        Base::Unload();
    }

    // Inner objects accessors
    ItemType& GetItem() { return Selector::m_item; }
    const ItemType& GetItem() const { return Selector::m_item; }
};

/** @}*/ //end of Device

}

#endif
