//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(EGL_EXTENSIONPROCEDURE_H)
#define EGL_EXTENSIONPROCEDURE_H


#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <CanderaPlatform/Device/Common/EGL/EglTraceMapper.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/Extensions/EglExtension.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/Extensions/EglExtensionProcedureResult.h>


namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */


/**
 * @brief Class encapsulating EGL extension procedure.
 */
template<typename Implementation, typename Result>
class EglExtensionProcedure
{
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaPlatformDevice);

public:
    // Construction

    EglExtensionProcedure(EglExtension& extension, const FeatStd::Char* name)
    :   m_extension(extension),
        m_name(name),
        m_procedure(0)
    { }


    // Interface

    template <typename Arguments>
    EglExtensionProcedureResult<Result> Call(const Arguments& arguments)
    {
        EglExtensionProcedureResult<Result> result;
        if (m_procedure == 0) {
            if (m_extension.IsSupported()) {
                void (*procedure)(void) = eglGetProcAddress(static_cast<const char*>(m_name));
                m_procedure = reinterpret_cast<Implementation>(procedure);
            } else {
                CANDERA_DEVICE_LOG_ERROR("EGL extension '%s' is not supported", m_extension.GetName());
            }
        }
        if (m_procedure != 0) {
            arguments.ApplyAndCall(m_procedure, result);
            if (result.IsErrorOrVoid()) {
                CANDERA_EGL_CHECK_ERROR_NO_RETURN(m_name);
            }
        } else {
            CANDERA_DEVICE_LOG_ERROR("Unable to obtain address of EGL procedure '%s' from extension '%s'",
                m_name, m_extension.GetName());
        }
        return result;
    }


private:
    FEATSTD_MAKE_CLASS_UNCOPYABLE(EglExtensionProcedure);


    // Attributes

    EglExtension& m_extension;
    const FeatStd::Char* m_name;
    Implementation m_procedure;
};


/** @}*/ //end of CommonDevice

} // namespace Candera


#endif // defined(EGL_EXTENSIONPROCEDURE_H)
