//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(EGL_SHARED_CONTEXT_PROVIDER_H)
#define EGL_SHARED_CONTEXT_PROVIDER_H

#include <CanderaPlatform/Device/Common/Base/ContextProvider3D.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/EglContextCreator.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief EglSharedContextProvider is a class used for sharing states. Such
 * class cannot be used as render target.
 * 
 * Instances of this class are full featured ContextProvider3D objects, as 
 * opposed to EglContextProvider which only represents a provider for a 
 * GduBaseRenderTarget.
 */
class EglSharedContextProvider : public ContextProvider3D
{
public:
    /**
     * Constructor.
     */
    EglSharedContextProvider() {}

    /**
     * Destructor.
     */
    virtual ~EglSharedContextProvider() { Unload(); }

    /**
     * Create the underlying context, and attach to the context resource pool.
     * @param display   Display used to create the context. This object will
     *                  keep a soft reference to the display while uploaded.
     *                  Unload this object before destroying the display.
     * @param contextResourcePool Pool to attach the context provider.
     * @return True if successful.
     */
    bool Upload(EGLDisplay display, ContextResourcePool& contextResourcePool);

    /**
     * Destroy the context.
     */
    FEATSTD_LINT_CLEANUP_FUNCTION(Candera::EglSharedContextProvider::Unload)
    bool Unload();

    // Interface of ContextProvider3D.

    /**
     *  Retrieves the context with a Shared type.
     *  Overrides method from RenderTarget3D.
     *  @return Pointer to native device context handle.
     */
    virtual void* GetContextHandle() const { return static_cast<void*>(ToEglChain().GetContext()); }
        
    /**
     *  Does nothing. Context may not be activated.
     *  Overrides method from RenderTarget3D.
     *  @return False.
     */
    virtual bool Activate() { return false; }

    /**
     *  Does nothing. Context may not be activated.
     *  Overrides method from RenderTarget.
     */
    virtual void BeginDraw() {}
        
    /**
     *  Does nothing.
     *  Overrides method from RenderTarget.
     */
    virtual void EndDraw() {}

    /**
     *  This context has no surface.
     *  Overrides method from RenderTarget.
     */
    virtual void SwapBuffers() {}

    /**
     *  This context has no surface.
     *  Overrides method from Surface.
     *  @return 0.
     */
    virtual Int GetHeight() const { return 0; }
        
    /**
     *  This context has no surface.
     *  Overrides method from Surface.
     *  @return 0.
     */
    virtual Int GetWidth() const { return 0; }

    /**
     *  Does nothing.
     *  Overrides method from Synchronizable.
     */
    virtual void Sync() {}

    /**
     *  Does nothing.
     *  Overrides method from Synchronizable.
     */
    virtual void WaitSync() {}

    // Interface of EglChain.
    const EglChain& ToEglChain() const { return m_context; }

private:
    class Context : public EglContextCreator
    {
    public:
        Context();
        ~Context() { Unload(); }

        /**
         * Create the underlying context.
         * @param display   Display used to create the context.
         * @param sharedContext   Context used to share state.
         * @return True if successful.
         */
        bool Upload(EGLDisplay display, EGLContext sharedContext);

        /**
         * Destroy the context.
         */
        FEATSTD_LINT_CLEANUP_FUNCTION(Candera::EglSharedContextProvider::Context::Unload)
        bool Unload();

        /**
         *  Retrieves the associated surface config.
         *  Overrides method from EglChain.
         *  @return handle of the associated EGLConfig.
         */
        virtual EGLConfig GetConfig() const { return m_config; }

        /**
         *  This context has no readable surface.
         *  Overrides method from EglChain.
         *  @return 0.
         */
        virtual EGLSurface GetReadSurface() const { return 0; }

        /**
         *  This context has no drawable surface.
         *  Overrides method from EglChain.
         *  @return 0.
         */
        virtual EGLSurface GetDrawSurface() const { return 0; }

        /**
         *  Retrieves the associated display.
         *  Overrides method from EglChain.
         *  @return handle of the associated EGLDisplay.
         */
        virtual EGLDisplay GetDisplay() const { return m_display; }

        /**
         *  Retrieves the base context used for resource sharing.
         *  Overrides virtual method from EglContextCreator.
         *  @return handle of the shared EGLContext.
         */
        virtual EGLContext GetShareContext() const { return m_sharedContext; }

        EGLDisplay m_display;
        EGLConfig m_config;
        EGLContext m_sharedContext;
    };

    Context m_context;
};

/** @}*/ //end of CommonDevice

}

#endif //EGL_SHARED_CONTEXT_PROVIDER_H
