//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <CanderaPlatform/Device/Common/Internal/EGL/EglSharedContextProvider.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>
#include <CanderaPlatform/Device/Common/EGL/EglTraceMapper.h>

namespace Candera
{

bool EglSharedContextProvider::Upload(EGLDisplay display, ContextResourcePool& contextResourcePool)
{
    bool success = m_context.Upload(display, 
        FeatStd::Internal::PointerToPointer<EGLContext>(contextResourcePool.GetSharedContextHandle()));

    if (!success) {
        CANDERA_DEVICE_LOG_WARN("Failed to upload context.");
        return false;
    }

    success = SetContextResourcePool(&contextResourcePool);
    // Render state cache not required since the context cannot be activated.

    return success;
}

bool EglSharedContextProvider::Unload()
{
    return (SetContextResourcePool(0) && m_context.Unload());
}

EglSharedContextProvider::Context::Context() :
    m_display(EGL_NO_DISPLAY),
    m_config(0),
    m_sharedContext(EGL_NO_CONTEXT)
{
}

bool EglSharedContextProvider::Context::Upload(EGLDisplay display, EGLContext sharedContext)
{
    if(m_display != EGL_NO_DISPLAY) {
        CANDERA_DEVICE_LOG_WARN("Already uploaded.");
        return true;
    }
    if(display == EGL_NO_DISPLAY) {
        CANDERA_DEVICE_LOG_WARN("No display.");
        return false;
    }

    EGLint numConfigs = 0;
    EGLBoolean success = eglGetConfigs(display, &m_config, 1, &numConfigs);
    CANDERA_DEVICE_CHECK_AND_LOG_WARN(Egl, "Failed to retrieve config");
    bool failedToRetrieveConfig =
        (success == EGL_FALSE) ||
        (numConfigs <= 0) ||
        CANDERA_DEVICE_PACKAGE_TRACE_CLASS(Egl)::WasLastErrorSet();
    if (failedToRetrieveConfig) {
        return false;
    }

    m_display = display;
    m_sharedContext = sharedContext;

    if (!EglContextCreator::Upload()) {
        m_display = EGL_NO_DISPLAY;
        m_sharedContext = EGL_NO_CONTEXT;

        CANDERA_DEVICE_LOG_WARN("Failed to upload context creator.");
        return false;
    }

    // Set shared context to current context to avoid dangling pointer.
    m_sharedContext = GetContext();

    return true;
}

bool EglSharedContextProvider::Context::Unload()
{
    if(m_display == EGL_NO_DISPLAY) {
        CANDERA_DEVICE_LOG_WARN("Already uploaded.");
        return true;
    }

    EglContextCreator::Unload();

    m_display = EGL_NO_DISPLAY;
    m_sharedContext = EGL_NO_CONTEXT;

    return true;
}

}
