//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(EGL_GENERIC_CONTEXT_PROVIDER_POOL_H)
#define EGL_GENERIC_CONTEXT_PROVIDER_POOL_H

#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#include <CanderaPlatform/Device/Common/Base/ContextResourcePool.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/EglGenericContextProvider.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief EglGenericContextProviderPool is a pool of EglGenericContextProvider.
 * 
 * Useful when implementing DevicePackageInterface::CreateDefaultContextProvider.
 * It attaches one EglGenericContextProvider per ContextResourcePool.
 */
class EglGenericContextProviderPool
{
    public:
        /**
         * Get the default generic context provider associated with the given display index.
         * @param index A display index.
         * @return A pointer to the context provider that should be used, or 0 if none is available.
         */
        static const EglGenericContextProvider* Get(Int index);

        /**
         * Get the default generic context provider associated with the given ContextResourcePool index.
         * @param index A ContextResourcePool index.
         * @return A pointer to the context provider that should be used, or 0 if none is available.
         */
        static EglGenericContextProvider* GetByPool(Int index);

        /**
         * Attach a new context provider. This has the same restrictions as 
         * DevicePackageInterface::CreateDefaultContextProvider.
         * @param pool          The context resource pool to which the context provider is attached.
         * @param display       A display handle.
         * @param config        A configuration object.
         * @param readSurface   A read surface handle.
         * @param drawSurface   A draw surface handle.
         * @param context       A context handle.
         */
        static void Attach(ContextResourcePool& pool, EGLDisplay display, EGLConfig config, EGLSurface readSurface, EGLSurface drawSurface, EGLContext context);

        /**
         * Detach an EglGenericContextProvider from the pool. If the object is not part of the pool, nothing happens.
         * @param provider      The provider that needs to be detached.
         */
        static void Detach(const EglGenericContextProvider* provider);

    private:
        typedef EglGenericContextProvider Pool[CANDERA_MAX_CONTEXT_COUNT];

        /// helper type for FEATSTD_UNSYNCED_STATIC_OBJECT
        struct PoolContainer {
            Pool mPool;
        };

        static Pool& GetPool();
};

/** @}*/ //end of CommonDevice

}

#endif
