//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(EGL_GENERIC_CONTEXT_PROVIDER_H)
#define EGL_GENERIC_CONTEXT_PROVIDER_H

#include <CanderaPlatform/Device/Common/Base/ContextProvider3D.h>
#include <CanderaPlatform/Device/Common/Internal/EGL/EglChain.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief EglGenericContextProvider is a class that attaches to the EGL object,
 * necessary for implementing a context provider, instead of creating them.
 * 
 * Instances of this class are full featured ContextProvider3D objects, a opposed to
 * EglContextProvider which only represents a provider for a GduBaseRenderTarget.
 */
class EglGenericContextProvider : public ContextProvider3D, public EglChain
{
    public:
        /**
         * Constructor. An empty context provider will have to receive all
         * attachments before being complete.
         */
        EglGenericContextProvider();

        /**
         * Destructor.
         */
        virtual ~EglGenericContextProvider();

        /**
         * Checks whether all attachments are applied. This doesn't check whether
         * the attachments are compatible. Activate could provide more information.
         * The Configuration object attached is only checked when other objects are created.
         * A list of checked attachments: display, read surface, draw surface, context,
         * context resource pool.
         * @return true if all attachments are present.
         */
        bool IsComplete() const;

        /**
         * Attach the handles to the object. Previously attached handles are detached.
         * The lifetime of the attached objects must be insured by the called of this function.
         * Other handles should be attached, or this object should be destroyed before their destruction.
         * Null handles can be attached resulting in an incomplete provider.
         * @param pool          The context resource pool to which this context provider is attached.
         * @param display       A display handle.
         * @param config        A configuration object.
         * @param readSurface   A read surface handle.
         * @param drawSurface   A draw surface handle.
         * @param context       A context handle.
         */
        void Attach(ContextResourcePool* pool, EGLDisplay display, EGLConfig config, EGLSurface readSurface, EGLSurface drawSurface, EGLContext context);

        /**
         * Convenience method to detach all handles from the context provider.
         */
        FEATSTD_LINT_CLEANUP_FUNCTION(Candera::EglGenericContextProvider::DetachAll)
        void DetachAll();

        // Interface of ContextProvider3D.

        /**
         *  Retrieves the context with a generic type.
         *  Overrides method from RenderTarget3D.
         *  @return Pointer to native device context handle.
         */
        virtual void* GetContextHandle() const { return static_cast<void*>(GetContext()); }
        
        /**
         *  Activates the context.
         *  Overrides method from RenderTarget3D.
         *  @return true upon success.
         */
        virtual bool Activate();

        /**
         *  Activates the EGL context. Overloads virtual method from GduContext.
         *  @return true if activation is successful
         */
        virtual bool ActivateContext();

        /**
         *  Sets up the context for drawing.
         *  Overrides method from RenderTarget.
         */
        virtual void BeginDraw() { Activate(); }
        
        /**
         *  Does nothing.
         *  Overrides method from RenderTarget.
         */
        virtual void EndDraw() {}

        /**
         *  Swaps the buffers of the draw surface.
         *  Overrides method from RenderTarget.
         */
        virtual void SwapBuffers();

        /**
         *  Retrieves the height of the draw surface.
         *  Overrides method from Surface.
         */
        virtual Int GetHeight() const;
        
        /**
         *  Retrieves the width of the draw surface.
         *  Overrides method from Surface.
         */
        virtual Int GetWidth() const;

        /**
         *  Does nothing.
         *  Overrides method from Synchronizable.
         */
        virtual void Sync() {}

        /**
         *  Does nothing.
         *  Overrides method from Synchronizable.
         */
        virtual void WaitSync() {}

        // Interface of EglChain.

        /**
         *  Retrieves the associated context.
         *  Overrides method from EglContext.
         *  @return handle of the associated EGLContext.
         */
        virtual EGLContext GetContext() const { return m_context; }

        /**
         *  Retrieves the associated surface config.
         *  Overrides method from EglChain.
         *  @return handle of the associated EGLConfig.
         */
        virtual EGLConfig GetConfig() const { return m_config; }

        /**
         *  Retrieves the surface used for reading operations.
         *  Overrides method from EglChain.
         *  @return handle of the associated EGLSurface.
         */
        virtual EGLSurface GetReadSurface() const { return m_readSurface; }

        /**
         *  Retrieves the surface used for writing operations.
         *  Overrides method from EglChain.
         *  @return handle of the associated EGLSurface.
         */
        virtual EGLSurface GetDrawSurface() const { return m_drawSurface; }

        /**
         *  Retrieves the associated display.
         *  Overrides method from EglChain.
         *  @return handle of the associated EGLDisplay.
         */
        virtual EGLDisplay GetDisplay() const { return m_display; }
    private:
        EGLDisplay m_display;
        EGLConfig m_config;
        EGLSurface m_readSurface;
        EGLSurface m_drawSurface;
        EGLContext m_context;
};

/** @}*/ //end of CommonDevice

}

#endif //EGL_GENERIC_CONTEXT_PROVIDER_H
