//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <CanderaPlatform/Device/Common/Internal/EGL/EglGenericContextProvider.h>
#include <CanderaPlatform/Device/Common/EGL/EglTraceMapper.h>

namespace Candera
{


EglGenericContextProvider::EglGenericContextProvider() :
    m_display(EGL_NO_DISPLAY),
    m_config(0),
    m_readSurface(EGL_NO_SURFACE),
    m_drawSurface(EGL_NO_SURFACE),
    m_context(EGL_NO_CONTEXT)
{
}

EglGenericContextProvider::~EglGenericContextProvider()
{
    DetachAll();
}

bool EglGenericContextProvider::IsComplete() const
{
    return (GetDisplay() != EGL_NO_DISPLAY) &&
        (GetReadSurface() != EGL_NO_SURFACE) &&
        (GetDrawSurface() != EGL_NO_SURFACE) &&
        (GetContext() != EGL_NO_CONTEXT) &&
        (GetContextResourcePool() != 0);
}

void EglGenericContextProvider::Attach(ContextResourcePool* pool, EGLDisplay display, EGLConfig config, EGLSurface readSurface, EGLSurface drawSurface, EGLContext context)
{
    DestroyRenderStateCache();
    static_cast<void>(SetContextResourcePool(0));

    m_display = display;
    m_config = config;
    m_readSurface = readSurface;
    m_drawSurface = drawSurface;
    m_context = context;

    static_cast<void>(SetContextResourcePool(pool));
    if (pool != 0) {
        static_cast<void>(CreateRenderStateCache());
    }
    if (IsComplete()) {
        static_cast<void>(Activate());
    }
}

void EglGenericContextProvider::DetachAll()
{
    Attach(0, EGL_NO_DISPLAY, 0, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
}


void EglGenericContextProvider::SwapBuffers()
{
    if ((m_display != EGL_NO_DISPLAY) && (m_drawSurface != EGL_NO_SURFACE)){
        NotifyListenersOnBeforeSwapBuffersCalled();
        static_cast<void>(eglSwapBuffers(m_display, m_drawSurface));
        NotifyListenersOnAfterSwapBuffersExecuted();
    }
}

Int EglGenericContextProvider::GetHeight() const
{
    EGLint value = 0;
    EGLBoolean success = eglQuerySurface(m_display, m_drawSurface, EGL_HEIGHT, &value);
    if (success == EGL_TRUE) {
        return value;
    }
    else {
        return 0;
    }
}
        
Int EglGenericContextProvider::GetWidth() const
{
    EGLint value = 0;
    EGLBoolean success = eglQuerySurface(m_display, m_drawSurface, EGL_WIDTH, &value);
    if (success == EGL_TRUE) {
        return value;
    }
    else {
        return 0;
    }
}

bool EglGenericContextProvider::Activate()
{
    return ActivateContext();
}

bool EglGenericContextProvider::ActivateContext()
{
    if (!IsComplete()) {
        CANDERA_DEVICE_LOG_INFO("incomplete");
        return false;
    }
    if (!EglChain::ActivateContext()) {
        CANDERA_DEVICE_LOG_WARN("base activation failed");
        return false;
    }

    ActivateContextResourcePool();
    ActivateRenderStateCache();

    return true;
}

}
