//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(EGL_CONTEXT_PROVIDER_H)
#define EGL_CONTEXT_PROVIDER_H

#include <CanderaPlatform/Device/Common/Base/ContextProvider3D.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageTrace.h>

#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>
#include <CanderaPlatform/Device/Common/Base/Display.h>

#include <CanderaPlatform/Device/Common/EGL/EglInclude.h>
#if defined(CGIDEVICE_OPENGLES_11) 
#include <CanderaPlatform/Device/Common/OpenGLES/GlInclude.h>
#endif
#include <CanderaPlatform/Device/Common/Internal/GDU/GduTools.h>

namespace Candera
{

/** @addtogroup CommonDevice
 *  @{
 */

/**
 * @brief EglContextProvider represents a wrapper that links a ContextProvider3D with an EglContext.
 *
 * Can be used as context in GduBaseRenderTarget.
 * As a ContextProvider3D, it is responsible with activating the current ContextResourcePool and
 * setting any render state cache attached to this Egl Context.
 */
template <
    typename TBase,
    typename TSupport,
    typename TProperties>
class EglContextProvider : public ContextProvider3D, public TBase
{
    CANDERA_LINT_DECLARE_CLEANUP_FUNCTION(Candera::EglContextProvider::Unload)

    FEATSTD_TYPEDEF_BASE(TBase);
    typedef TSupport Support;
    typedef TProperties Properties;

    public:

        /**
         *  Constructs a EglContextProvider object.
         */
        EglContextProvider();

        /**
         *  Uploads this object to video memory.
         *  @param displayId         Display to which this render target is attached.
         *  @param support           Passed to Base.
         *  @param properties        Passed to Base.
         *  @return True if successful, false otherwise.
         */
        bool Upload(Int displayId, Support& support, Properties& properties);

        /**
         *  Unloads this object from video memory.
         */
        void Unload();

        /**
         *  Activates the associated context and sets up render state caching and
         *  the context resource pool.
         *  @return true if activation is successful
         */
        virtual bool Activate();

        /**
         *  Activates the EGL context. Overloads virtual method from GduContext.
         *  @return true if activation is successful
         */
        virtual bool ActivateContext();

        void* GetContextHandle() const override {
            return static_cast<void*>(Base::GetContextHandle());
        }

        using RenderTarget::NotifyListenersOnBeforeSwapBuffersCalled;
        using RenderTarget::NotifyListenersOnAfterSwapBuffersExecuted;
        
    protected:
        /**
         *  Retrieves the base context used for resource sharing. Overrides virtual method from EglContextCreator.
         *  @return handle of the shared EGLContext.
         */
        virtual EGLContext GetShareContext() const;

        /**
         *  Non-virtual. An object should never be destroyed from a pointer to this class.
         */
        ~EglContextProvider();

    private:
        bool m_isUploaded;
        bool m_mustRemovePool;
        EGLContext m_sharedContext;
};

/** @}*/ //end of CommonDevice
template <typename TBase, typename TSupport, typename TProperties>
EglContextProvider<TBase, TSupport, TProperties>::EglContextProvider() : 
    m_isUploaded(false),
    m_mustRemovePool(false),
    m_sharedContext(EGL_NO_CONTEXT)
{
}

template <typename TBase, typename TSupport, typename TProperties>
EglContextProvider<TBase, TSupport, TProperties>::~EglContextProvider() 
{
    m_mustRemovePool = true;
    Unload();
}

template <typename TBase, typename TSupport, typename TProperties>
bool EglContextProvider<TBase, TSupport, TProperties>::ActivateContext()
{
    // Only activate context resource pool if base upload is completed.
    if (m_sharedContext == EGL_NO_CONTEXT) {
        if (!m_isUploaded) {
            return false;
        }
    }
    if (!Base::ActivateContext()) {
        CANDERA_DEVICE_LOG_WARN("base activation failed");
        return false;
    }

    if (m_sharedContext == EGL_NO_CONTEXT) {
        ActivateContextResourcePool();
        ActivateRenderStateCache();
    }

    return true;
}

template <typename TBase, typename TSupport, typename TProperties>
bool EglContextProvider<TBase, TSupport, TProperties>::Activate()
{
    //Activate only works when EglContextProvider is completely uploaded.
    if (!m_isUploaded) {
        return false;
    }
    return ActivateContext();
}

template <typename TBase, typename TSupport, typename TProperties>
bool EglContextProvider<TBase, TSupport, TProperties>::Upload(Int displayId, Support& support, Properties& properties)
{
    if (m_isUploaded) { 
        return true;
    }
    ContextResourcePool* pool = GetContextResourcePool();
    if (pool == 0) {
        pool = Internal::GduTools::GetContextResourcePoolByDisplayId(displayId);
        m_mustRemovePool = true;
    }

    if (pool == 0) {
        CANDERA_DEVICE_LOG_WARN("no context resource pool");
        return false;
    }

    SetContextResourcePool(pool);
    CreateRenderStateCache();

    m_sharedContext = static_cast<EGLContext>(pool->GetSharedContextHandle());

    if (!Base::Upload(displayId, support, properties)) {
        CANDERA_DEVICE_LOG_WARN("base upload failed");
        Unload();
        return false;
    }

    m_sharedContext = EGL_NO_CONTEXT;
    m_isUploaded = true;

    if (!this->ActivateContext()) {
        CANDERA_DEVICE_LOG_WARN("base upload failed");
        Unload();
        m_isUploaded = false;
        return false;
    }
#if defined(CGIDEVICE_OPENGLES_11) 
    glEnable (GL_LIGHTING);
#ifdef FEATSTD_GLERRORS_AND_EGLERRORS_ENABLED
    UInt32 glError = static_cast<UInt32>(glGetError());
    if (GL_NO_ERROR != glError) {
        CANDERA_DEVICE_LOG_WARN("glEnable(GL_LIGHTING) failed for current context.");
    }
#endif
    glEnable(GL_RESCALE_NORMAL);
    glEnable(GL_NORMALIZE);
#ifdef FEATSTD_GLERRORS_AND_EGLERRORS_ENABLED
    glError = static_cast<UInt32>(glGetError());
    if (GL_NO_ERROR != glError) {
            CANDERA_DEVICE_LOG_WARN("glEnable(GL_RESCALE_NORMAL) failed for current context.");
    }
#endif
#endif

    return true;
}

template <typename TBase, typename TSupport, typename TProperties>
void EglContextProvider<TBase, TSupport, TProperties>::Unload()
{
    if (!m_isUploaded) {
        return;
    }

    // Callback to Renderer::OnContextUnload
    OnContextUnload();

    Base::Unload();
    DestroyRenderStateCache();
    if (m_mustRemovePool) {
        m_mustRemovePool = false;
        SetContextResourcePool(0);
    }
    m_isUploaded = false;
}

template <typename TBase, typename TSupport, typename TProperties>
EGLContext EglContextProvider<TBase, TSupport, TProperties>::GetShareContext() const
{
    if (m_sharedContext != EGL_NO_CONTEXT) {
        return m_sharedContext;
    }
    else {
        return static_cast<EGLContext>(GetSharedContextHandle());
    }
}

}

#endif
