//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ALLOCATOR_BASE_H)
#define CANDERA_ALLOCATOR_BASE_H

#include <Candera/Environment.h>
#include <CanderaPlatform/Device/Common/Internal/BlockContainer.h>

namespace Candera {
    /**
     * @brief Allocator is a helper class intended to handle generic tasks involved in
     * target buffer allocation.
     *
     * @param TInfo Defines the platform specific type of the stored surface data.
     * @param THandle Defines the platform specific type of the stored surface data.
     **/
    template<typename TInfo, typename THandle>
    class AllocatorBase
    {
        struct Wrapper;
        typedef Internal::BlockContainer<Wrapper> BlockContainer;
        typedef typename BlockContainer::Iterator Iterator;
    public:

        virtual ~AllocatorBase()
        {
            for (Iterator it = m_container.Begin(); it != m_container.End(); it++) {
                if (it->allocated) {
                    it->object.Clear();
                }
            }
            m_container.Clear();
        }

        bool CreateItems(Int count, THandle *pSurface)
        {
            Iterator it = m_container.Begin();
            while ((count > 0) && (it != m_container.End())) {
                if (!it->allocated) {
                    it->allocated = true;
                    pSurface[-- count] = GetItemHandleFromWrapper(&*it);
                }
                ++it;
            }

            if (count > 0) {
                it = m_container.Append(count);
                while (it != m_container.End()) {
                    it->allocated = true;
                    pSurface[-- count] = GetItemHandleFromWrapper(&*it);
                    ++it;
                }
            }

            return count == 0;
        }

        bool DestroyItems(Int count, THandle *pSurface)
        {
            bool success = true;
            while (count > 0){
                Wrapper* wrapper = GetWrapperFromItemHandle(pSurface[-- count]);
                pSurface[count] = 0;
                if (wrapper == 0) {
                    success = false;
                    continue;
                }

                wrapper->allocated = false;

                if (!wrapper->object.Clear()){
                    success = false;
                    continue;
                }
            }

            SizeType current = 0;
            SizeType highest = 0;
            for (Iterator it = m_container.Begin(); it != m_container.End(); it++) {
                if (it->allocated) {
                    highest = current;
                }
                current ++;
            }

            if (highest * 2 <= m_container.Size()) {
                m_container.Resize(highest + 1);
                m_container.Reserve(highest + 1);
            }

            return success;
        }

        static TInfo* GetDataFromItemHandle(THandle surface)
        {
            Wrapper* wrapper = GetWrapperFromItemHandle(surface);
            return (wrapper == 0) ? 0 : &wrapper->object;
        }

    private:
        static Wrapper* GetWrapperFromItemHandle(THandle surface)
        {
            Wrapper* wrapper = reinterpret_cast<Wrapper*>(surface);
            return ((wrapper != 0) && (wrapper->allocated)) ? wrapper : 0;
        }

        static THandle GetItemHandleFromWrapper(const Wrapper* wrapper)
        {
            return (wrapper != 0) ? reinterpret_cast<THandle>(wrapper) : 0;
        }

        struct Wrapper
        {
            bool allocated;
            TInfo object;

            Wrapper() : allocated(false), object() {}
        };

        BlockContainer m_container;
    };
} // namespace Candera
#endif //CANDERA_ALLOCATOR_BASE_H
