//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextBrushProperties.h"

#include <Candera/System/GlobalizationBase/CultureManager.h>
#include <FeatStd/Util/FeatLimits.h>

namespace Candera {
    using namespace TextRendering;
    namespace Internal {

        /******************************************************************************
         *  Constructor
         ******************************************************************************/
        TextBrushProperties::TextBrushProperties() :
            m_complexScriptMode(TextBrushComplexScriptDisabled),
            m_cacheType(GlyphCache),
            m_text(),
            m_style(),
            m_currentStyleVersion(0),
            m_culture(),
            m_color(Candera::Color(1.0F, 1.0F, 1.0F, 1.0F)),
            m_horizontalAlignment(),
            m_verticalAlignment(),
            m_glyphSpacing(0.F),
            m_lineSpacing(1.F),
            m_pixelwiseLineSpacing(false),
            m_multiLineLayouting(true),
            m_wordWrap(false),
            m_layoutStrategy(0)
            
        {
        }

        TextBrushProperties::TextBrushProperties(const TextBrushProperties& rhs) :
            m_complexScriptMode(rhs.m_complexScriptMode.Get()),
            m_cacheType(rhs.m_cacheType.Get()),
            m_text(),   // see special handling for font in function scope
            m_style(rhs.m_style.Get()),
            m_currentStyleVersion(rhs.m_currentStyleVersion.Get()),
            m_culture(rhs.m_culture.Get()),
            m_layoutingArea(rhs.m_layoutingArea.Get()),
            m_cacheArea(rhs.m_cacheArea.Get()),
            m_color(rhs.m_color.Get()),
            m_horizontalAlignment(rhs.m_horizontalAlignment.Get()),
            m_verticalAlignment(rhs.m_verticalAlignment.Get()),
            m_glyphSpacing(rhs.m_glyphSpacing.Get()),
            m_lineSpacing(rhs.m_lineSpacing.Get()),
            m_pixelwiseLineSpacing(rhs.m_pixelwiseLineSpacing.Get()),
            m_multiLineLayouting(rhs.m_multiLineLayouting.Get()),
            m_wordWrap(rhs.m_wordWrap.Get()),
            m_layoutStrategy(rhs.m_layoutStrategy.Get())
        {
            if (rhs.m_text.GetDisposerFn() != 0) {     // the original text has a disposer function, so a copy is needed!
                SizeType length = StringPlatform::Length(rhs.m_text.Get());
                if (length > 0) {
                    TChar* buffer = CANDERA_NEW_ARRAY(TChar, length + 1);
                    if (buffer != 0) {
                        StringPlatform::Copy(buffer, rhs.m_text);
                        m_text.Set(buffer, MemoryManagement::ArrayDisposer<const TChar*>::Dispose);
                    }
                }
            }
            else {
                m_text.Set(rhs.m_text, 0);
            }
        }

PixelSize TextBrushProperties::GetLineSpacing() const
{
    if (m_style() != 0) {
        Float size = static_cast<Float>((m_pixelwiseLineSpacing()?(static_cast<Float>(1.0F)):(static_cast<Float>(m_style()->GetMetrics().lineHeight))) * m_lineSpacing());
        

        bool sizeFits = (size > 0.F) && 
            (size < static_cast<Float>(FeatStd::Internal::Limits<PixelSize>::Max()));
        if (sizeFits){
            return static_cast<PixelSize>(size);
        }
    }
    return 0;
}

LayoutingOptions TextBrushProperties::GetLayoutingOptions() const
{
    LayoutingOptions options(
        m_horizontalAlignment,
        m_verticalAlignment,
        TextCoordinate(),
        TextSize(
            static_cast<Int16>(m_layoutingArea.Get().GetX()),
            static_cast<Int16>(m_layoutingArea.Get().GetY())),
        m_multiLineLayouting,
        m_wordWrap,
        GetLineSpacing(),
        static_cast<PixelPosition>(m_glyphSpacing.Get()));
    options.SetLayoutStrategy(m_layoutStrategy);
    return options;
}

ShapingOptions TextBrushProperties::GetShapingOptions() const
{
    Globalization::Culture::SharedPointer culture = m_culture();
    if (culture.PointsToNull()) {
        culture = Globalization::CultureManager::GetInstance().GetCurrentCulture();
    }
    ShapingOptions shapingOptions(m_style());
    shapingOptions.BindCulture(culture);

    return shapingOptions;
}
TextProperties TextBrushProperties::GetTextProperties() const
{
    if(!m_preprocessedText.Get().IsValid()) {
        return TextProperties(m_text.Get());
    }
    else {
        return TextProperties(m_preprocessedText.Get());
    }
}

    }   // namespace Internal
}   // namespace Candera

