//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SurfaceTextRenderContext.h"

#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera {

SurfaceTextRenderContext::SurfaceTextRenderContext():
    m_context(0),
    m_clip(-4096, -4096, 4096, 4096) 
{
}
void SurfaceTextRenderContext::Set2DContext(Handle context2D) {
    m_context = context2D;
}
const TextRendering::TextRect& SurfaceTextRenderContext::GetClipRect() const {
    if (m_context != 0){
        Float left;
        Float top;
        Float width;
        Float height;
        if (RenderDevice2D::GetActiveArea(m_context, RenderDevice2D::DestinationSurface, &left, &top, &width, &height)) {

            Matrix3x2 trans(m_transformation);
            trans.Inverse();

            Rectangle viewport(0.0F, 0.0F, width, height);
            viewport.Transform(trans);

            m_clip = TextRendering::TextRect(
                TextRendering::TextCoordinate(
                    static_cast<Int16>(Math::Floor(viewport.GetLeft())), 
                    static_cast<Int16>(Math::Floor(viewport.GetTop()))), 
                TextRendering::TextSize(
                    static_cast<Int16>(Math::Ceil(viewport.GetWidth())), 
                    static_cast<Int16>(Math::Ceil(viewport.GetHeight()))));
        }
    }
    return m_clip;
}

void SurfaceTextRenderContext::Blit(Int16 x, Int16 y, const TextRendering::GlyphBitmap &glyph)
{
    CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "Blit"));
    if (m_context == 0) {
        return;
    }
    if ((glyph.pixels == 0) || (glyph.width == 0) || (glyph.height == 0)) {
        return;
    }

    SurfaceHandle texture;
    bool created = RenderDevice2D::CreateSurfaces(1, &texture);
    bool result = created && RenderDevice2D::CreateBuffer(
        texture,
        glyph.width,
        glyph.height,
        glyph.pitch,
        glyph.format,
        glyph.pixels);

    if (result) {
        Matrix3x2 pos(m_transformation);
        pos.Translate(static_cast<Float>(x), static_cast<Float>(y));

        result = Renderer2D::SetTransformationMatrix(m_context, RenderDevice2D::SourceSurface, pos);
        result = result && RenderDevice2D::SetSurface(m_context, RenderDevice2D::SourceSurface, texture);
        result = result && Renderer2D::Blit(m_context);

        if (result) {
            Float left;
            Float top;
            Float width;
            Float height;
            if (RenderDevice2D::GetUpdatedArea(m_context, &left, &top, &width, &height)) {
                m_updatedArea.Union(Rectangle(left, top, width, height));
            }
        }
    }

    if (created) {
        static_cast<void>(RenderDevice2D::DestroySurfaces(1, &texture));
    }
}

}   // namespace Candera
