//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SurfaceTextBrushCache.h"

#include <Candera/Engine2D/Core/Renderer2D.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnitAttributes.h>
#include <CanderaPlatform/Device/Common/Base/ImageSource2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>

namespace Candera {
    using namespace TextRendering;
    namespace Internal {

        FEATSTD_RTTI_DEFINITION(SurfaceTextBrushCache, TextBrushCache)

        /******************************************************************************
         *  Constructor
         ******************************************************************************/
        SurfaceTextBrushCache::SurfaceTextBrushCache() :
            m_gdu()
        {
        }

        /******************************************************************************
         *  Destructor
         ******************************************************************************/
        SurfaceTextBrushCache::~SurfaceTextBrushCache()
        {
            Finalize();
        }

        /******************************************************************************
         *  Update
         ******************************************************************************/
        bool SurfaceTextBrushCache::Update(const TextBrushProperties& textBrushProperties, const Rectangle& textRect)
        {
            TextRenderer textRenderer;

            if (textBrushProperties.m_style().GetPointerToSharedInstance() == 0) {
                Finalize();
                return true;
            }

            if ((textBrushProperties.m_text() == 0) ||
                (textRect.GetHeight() <= Float(0)) ||
                (textRect.GetWidth() <= Float(0)))
            {
                Finalize();
                return true;
            }

            //text bound
            Int16 textLeft = static_cast<Int16>( textRect.GetLeft());
            Int16 textTop = static_cast<Int16>( textRect.GetTop());

            // Determine if surface needs to be created.
            bool createGdu = (m_gdu == 0) || (m_gdu->ToImageSource2D() == 0);

            Int surfaceWidth = static_cast<Int>( textRect.GetWidth());
            if (textBrushProperties.m_cacheArea().GetX() > 0.F) {
                surfaceWidth = static_cast<Int>(textBrushProperties.m_cacheArea().GetX());
                // Create surface if it does not match the requested size.
                createGdu = createGdu ||
                    (m_gdu->ToImageSource2D()->GetWidth() != surfaceWidth);
            }
            else {
                // Create surface if it can not hold the current text.
                createGdu = createGdu ||
                    (m_gdu->ToImageSource2D()->GetWidth() < surfaceWidth);
            }
            Int surfaceHeight = static_cast<Int>( textRect.GetHeight());
            if (textBrushProperties.m_cacheArea().GetY() > 0.F) {
                surfaceHeight = static_cast<Int>(textBrushProperties.m_cacheArea().GetY());
                // Create surface if it does not match the requested size.
                createGdu = createGdu ||
                    (m_gdu->ToImageSource2D()->GetHeight() != surfaceHeight);
            }
            else {
                // Create surface if it can not hold the current text.
                createGdu = createGdu ||
                    (m_gdu->ToImageSource2D()->GetHeight() < surfaceHeight);
            }

            // Don't generate gdu if the size is 0.
            if (surfaceWidth * surfaceHeight == 0) {
                Finalize();
                return true;
            }

            // Create surface if it should be shrank at least two times.
            createGdu = createGdu ||
                ((m_gdu->ToImageSource2D()->GetWidth() * m_gdu->ToImageSource2D()->GetHeight()) > (2 * (surfaceWidth * surfaceHeight)));

            //clean old cached surface
            if ((m_gdu != 0) && createGdu) {
                Finalize();
            }

            if (surfaceWidth * surfaceHeight <= 0) {
                return true;
            }

            //create new cache surface
            if (m_gdu == 0) {
                //TO DO: 2D objects should be aware of which display they are uploaded to.
                const GraphicDeviceUnitTypeHandle *type;
                Int count;
                type = DevicePackageDescriptor::GetUnitTypes(
                    DevicePackageDescriptor::OffscreenTarget2D,
                    0,
                    count);

                if ((type == 0) || (count < 1)) {
                    return false;
                }

                const Int gduConfiguration[] = {
                    static_cast<Int>(GduWidthAttr), surfaceWidth,
                    static_cast<Int>(GduHeightAttr), surfaceHeight,
                    static_cast<Int>(GduRedSizeAttr), 0,
                    static_cast<Int>(GduGreenSizeAttr), 0,
                    static_cast<Int>(GduBlueSizeAttr), 0,
#ifdef CANDERA_4BIT_GLYPH_CACHE_ENABLED
                    static_cast<Int>(GduAlphaSizeAttr), 4,
#else
                    static_cast<Int>(GduAlphaSizeAttr), 8,
#endif
                    static_cast<Int>(GduAttributeListEnd)};

                m_gdu = DevicePackageInterface::CreateGraphicDeviceUnit(type[0], &gduConfiguration[0]);
                if (m_gdu == 0) {
                    return false;
                }

                bool uploadAndValidate =
                    m_gdu->Upload() &&
                    (m_gdu->ToImageSource2D() != 0) &&
                    (m_gdu->ToRenderTarget2D() != 0);

                if (!uploadAndValidate) {
                    Finalize();
                    return false;
                }
            }

            m_gdu->ToRenderTarget2D()->BeginDraw();
            ContextHandle2D output = m_gdu->ToRenderTarget2D()->Get2DContextHandle();

            bool result = Renderer2D::Clear(output, 0.0F, 0.0F, 0.0F, 0.0F);

            if (result) {
            m_textRenderContext.ResetUpdatedArea();
            m_textRenderContext.Set2DContext(output);
            m_textRenderContext.SetTransformation(Matrix3x2());

                result = RenderDevice2D::SetBlendOperation(output,
                    RenderDevice2D::SourceSurface,
                    RenderDevice2D::Add,
                    RenderDevice2D::One,
                    RenderDevice2D::One);
                LayoutingOptions layoutingOptions = textBrushProperties.GetLayoutingOptions();
                layoutingOptions.SetOffset(TextCoordinate(-textLeft, -textTop));
                result = result && textRenderer.Render(
                    m_textRenderContext,
                    layoutingOptions,
                    textBrushProperties.GetShapingOptions(),
                    textBrushProperties.GetTextProperties());
            }

            m_gdu->ToRenderTarget2D()->EndDraw();
            m_gdu->ToRenderTarget2D()->SwapBuffers();

            if (!result) {
                Finalize();
                return false;
            }

            return true;
        }

        /******************************************************************************
         *  Render
         ******************************************************************************/
        void SurfaceTextBrushCache::Render(const TextBrushProperties& /*textBrushProperties*/, const Matrix3x2& transform, const Rectangle& bound, ContextHandle2D output, Rectangle& outputArea)
        {
            if (m_gdu != 0) {
                Matrix3x2 correctedTransform(transform);
                correctedTransform.Translate(bound.GetLeft(), bound.GetTop());
                bool result = Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, correctedTransform);

                result = result && RenderDevice2D::SetSurface(output, RenderDevice2D::SourceSurface, m_gdu->ToImageSource2D()->GetSurfaceHandle());
                result = result && RenderDevice2D::SetActiveArea(
                    output,
                    RenderDevice2D::SourceSurface,
                    0.0F,
                    0.0F,
                    bound.GetWidth(),
                    bound.GetHeight());
                result = result && Renderer2D::Blit(output);

                if (result) {
                    Float left;
                    Float top;
                    Float width;
                    Float height;
                    if (RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height)) {
                        outputArea = Rectangle(left, top, width, height);
                    }
                }
            }
        }

        void SurfaceTextBrushCache::Finalize()
        {
            if (m_gdu != 0) {
                m_gdu->Unload();
                DevicePackageInterface::DestroyGraphicDeviceUnit(m_gdu);
                m_gdu = 0;
            }
        }

    }   // namespace Internal
}   // namespace Candera
