//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SurfaceCacheTextRenderContext.h"
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>
#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>

namespace Candera {

UInt8* SurfaceGlyphCacheAccess::Create(const TextRendering::GlyphBitmap &bitmap, UInt8* cacheItem)
{
    SurfaceHandle texture = FeatStd::Internal::PointerToScalar<SurfaceHandle>(cacheItem);

    if ((bitmap.pixels != 0) && (bitmap.width != 0) && (bitmap.height != 0)) {
        bool isCreateBufferRequired = (texture == 0);
        if (isCreateBufferRequired) {
            static_cast<void>(RenderDevice2D::CreateSurfaces(1, &texture));
        }
        if (isCreateBufferRequired || (!Internal::RenderDevice2DHelper::HasBuffer(texture))) {
            static_cast<void>(RenderDevice2D::CreateBuffer(
                texture,
                bitmap.width,
                bitmap.height,
                bitmap.pitch,
                bitmap.format,
                bitmap.pixels));
        }
    }

    return FeatStd::Internal::ScalarToPointer<UInt8*>(texture);
}
void SurfaceGlyphCacheAccess::DestroyCacheItem(TextRendering::PixelBuffer buffer)
{
    SurfaceHandle texture = FeatStd::Internal::PointerToScalar<SurfaceHandle>(buffer);
    static_cast<void>(RenderDevice2D::DestroySurfaces(1, &texture));
}

bool SurfaceGlyphCacheAccess::CheckCacheItem(TextRendering::PixelBuffer buffer)
{
    SurfaceHandle texture = FeatStd::Internal::PointerToScalar<SurfaceHandle>(buffer);
    return (texture != 0) && (Internal::RenderDevice2DHelper::HasBuffer(texture));
}

SurfaceCacheTextRenderContext::SurfaceCacheTextRenderContext():
    m_context(0),
    m_clip(-4096, -4096, 4096, 4096)
{
}
void SurfaceCacheTextRenderContext::Set2DContext(Handle context2D)
{
    m_context = context2D;
}
Handle SurfaceCacheTextRenderContext::Get2DContext() const
{
    return m_context;
}
const TextRendering::TextRect& SurfaceCacheTextRenderContext::GetClipRect() const
{
    if (m_context != 0) {
        Float left;
        Float top;
        Float width;
        Float height;
        if (RenderDevice2D::GetActiveArea(m_context, RenderDevice2D::DestinationSurface, &left, &top, &width, &height)) {

            Matrix3x2 trans(m_transformation);
            trans.Inverse();

            Rectangle viewport(0.0F, 0.0F, width, height);
            viewport.Transform(trans);

            m_clip = TextRendering::TextRect(
                TextRendering::TextCoordinate(
                    static_cast<Int16>(Math::Floor(viewport.GetLeft())), 
                    static_cast<Int16>(Math::Floor(viewport.GetTop()))), 
                TextRendering::TextSize(
                    static_cast<Int16>(Math::Ceil(viewport.GetWidth())), 
                    static_cast<Int16>(Math::Ceil(viewport.GetHeight()))));
        }
    }
    return m_clip;
}

void SurfaceCacheTextRenderContext::Blit(Int16 x, Int16 y, const TextRendering::GlyphBitmap &glyph)
{
    if (m_context == 0) {
        return;
    }

    SurfaceHandle texture = FeatStd::Internal::PointerToScalar<SurfaceHandle>(glyph.pixels);
    if (texture == 0) {
        return;
    }

    Matrix3x2 pos(m_transformation);
    pos.Translate(static_cast<Float>(x), static_cast<Float>(y));

    bool result = Renderer2D::SetTransformationMatrix(m_context, RenderDevice2D::SourceSurface, pos);
    result = result && RenderDevice2D::SetSurface(m_context, RenderDevice2D::SourceSurface, texture);
    result = result && Renderer2D::Blit(m_context);

    if (result) {
        Float left;
        Float top;
        Float width;
        Float height;
        if (RenderDevice2D::GetUpdatedArea(m_context, &left, &top, &width, &height)) {
            m_updatedArea.Union(Rectangle(left, top, width, height));
        }
    }
}

/******************************************************************************
*  GetSurfaceGlyphCacheAccess
******************************************************************************/
SurfaceGlyphCacheAccess& SurfaceCacheTextRenderContext::GetSurfaceGlyphCacheAccess()
{
    FEATSTD_SYNCED_STATIC_OBJECT(SurfaceGlyphCacheAccess, instance);
    return instance;
}

}   // namespace Candera
