//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BitmapTextBrushCache.h"
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/EngineBase/Common/BitmapGlyphCacheAccess.h>
#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaPlatformDevice);
    using namespace TextRendering;
    namespace Internal {
        FEATSTD_RTTI_DEFINITION(BitmapTextBrushCache, TextBrushCache)

        /******************************************************************************
         *  Constructor
         ******************************************************************************/
        BitmapTextBrushCache::BitmapTextBrushCache()
            : m_bitmapImage(0)
        {
        }

        /******************************************************************************
         *  Destructor
         ******************************************************************************/
        BitmapTextBrushCache::~BitmapTextBrushCache()
        {
        }

        class GlyphCacheAccessTextRenderContext : public TextRendering::TextRenderContext
        {
        public:
            GlyphCacheAccessTextRenderContext()
            {
        }

            virtual const TextRect& GetClipRect() const
            {
                return m_textRect;
            }

            virtual void Blit(Int16, Int16, const GlyphBitmap &)
            {
            }

            virtual TextRendering::GlyphCacheAccess* GetGlyphCacheAccess() const {
                return &BitmapGlyphCacheAccess::GetInstance();
            }
            TextRect m_textRect;
        };


        static const TextRendering::TextRenderContext& BitmapTextNodeRendererGetMeasureReferenceContext()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(GlyphCacheAccessTextRenderContext, s_glyphCacheAccessTextRenderContext);
            return s_glyphCacheAccessTextRenderContext;
        }

        /******************************************************************************
         *  GetTextRenderContext
        ******************************************************************************/
        const TextRendering::TextRenderContext* BitmapTextBrushCache::GetTextRenderContext() const
        {
            return &BitmapTextNodeRendererGetMeasureReferenceContext();
        }

        /******************************************************************************
         *  Update
         ******************************************************************************/
        bool BitmapTextBrushCache::Update(const TextBrushProperties& textBrushProperties, const Rectangle& textRect)
        {
            BitmapTextRenderContext textRenderContext(GetTextRenderContext());
            TextRenderer textRenderer;

            if (textBrushProperties.m_style().PointsToNull()) {
                m_bitmapImage.Release();
                return true;
            }

            if ((textBrushProperties.m_text() == 0) || 
                (textRect.GetHeight() <= Float(0)) || 
                (textRect.GetWidth() <= Float(0)))
            {
                m_bitmapImage.Release();
                return true;
            }

            //text bound
            Int16 textLeft = static_cast<Int16>( textRect.GetLeft());
            Int16 textTop = static_cast<Int16>( textRect.GetTop());

            // Determine if bitmap needs to be created.
            // Create bitmap if it does not exist.
            bool createBitmap = m_bitmapImage.PointsToNull();

            UInt16 bitmapWidth = static_cast<UInt16>(textRect.GetWidth());
            if (bitmapWidth > FeatStd::Internal::Limits<UInt16>::Max()) {
                FEATSTD_LOG_WARN("Text width is to big for Bitmap caching. Text might be cut.");
                bitmapWidth = FeatStd::Internal::Limits<UInt16>::Max();
            }
            if (textBrushProperties.m_cacheArea().GetX() > 0.F) {
                if (static_cast<UInt16>(textBrushProperties.m_cacheArea().GetX()) > FeatStd::Internal::Limits<UInt16>::Max()) {
                    FEATSTD_LOG_WARN("Cache set width is to big for Bitmap caching. Text might be cut.");
                    bitmapWidth = FeatStd::Internal::Limits<UInt16>::Max();
                }
                else {
                    bitmapWidth = static_cast<UInt16>(textBrushProperties.m_cacheArea().GetX());
                }
                // Create bitmap if it does not match the requested size.
                createBitmap = createBitmap ||
                    (m_bitmapImage->GetWidth() != bitmapWidth);
            }
            else {
                // Create bitmap if it can not hold the current text.
                createBitmap = createBitmap ||
                    (m_bitmapImage->GetWidth() < bitmapWidth);
            }
            UInt16 bitmapHeight = static_cast<UInt16>( textRect.GetHeight());
            if (bitmapHeight > FeatStd::Internal::Limits<UInt16>::Max()) {
                FEATSTD_LOG_WARN("Text height is to big for Bitmap caching. Text might be cut.");
                bitmapHeight = FeatStd::Internal::Limits<UInt16>::Max();
            }
            if (textBrushProperties.m_cacheArea().GetY() > 0.F) {
                if (static_cast<UInt16>(textBrushProperties.m_cacheArea().GetY()) > FeatStd::Internal::Limits<UInt16>::Max()) {
                    FEATSTD_LOG_WARN("Cache set height is to big for Bitmap caching. Text might be cut.");
                    bitmapHeight = FeatStd::Internal::Limits<UInt16>::Max();
                }
                else {
                    bitmapHeight = static_cast<UInt16>(textBrushProperties.m_cacheArea().GetY());
                }
                // Create bitmap if it does not match the requested size.
                createBitmap = createBitmap ||
                    (m_bitmapImage->GetHeight() != bitmapHeight);
            }
            else {
                // Create bitmap if it can not hold the current text.
                createBitmap = createBitmap ||
                    (m_bitmapImage->GetHeight() < bitmapHeight);
            }

            // Don't generate bitmap if the size is 0.
            if (bitmapWidth * bitmapHeight == 0) {
                m_bitmapImage.Release();
                return true;
            }

            // Create bitmap if it should be shrank at least two times.
            createBitmap = createBitmap ||
                ((m_bitmapImage->GetWidth() * m_bitmapImage->GetHeight()) > static_cast<UInt32>((2U * (bitmapWidth * bitmapHeight))));

            //create a new bitmap
            if (createBitmap) {
                UInt8* pixels = FEATSTD_NEW_ARRAY(UInt8, static_cast<SizeType>(bitmapWidth) * static_cast<SizeType>(bitmapHeight));
                if (pixels == 0) {
                    return false;
                }

                Bitmap::SharedPointer bitmap = Bitmap::Create(
                    bitmapWidth, bitmapHeight,
                    Bitmap::AlphaUnsignedBytePixelFormat, Bitmap::PackAlignment1,
                    pixels, Bitmap::Disposer::Dispose);

                if (bitmap == 0) {
                    FEATSTD_DELETE_ARRAY(pixels);
                    return false;
                }

                m_bitmapImage = BitmapImage2D::Create();
                if (m_bitmapImage.PointsToNull() || (!m_bitmapImage->SetBitmap(bitmap))) {
                    return false;
                }
            }

            //unload the old bitmap
            if (m_bitmapImage->IsUploaded()) {
                bool unloadResult = m_bitmapImage->Unload();
                if ((!unloadResult) || m_bitmapImage->IsUploaded()) {
                    return false;
                }
            }
            //update bitmap
            const Bitmap::SharedPointer& bitmap = m_bitmapImage->GetMutableBitmap();
            Bitmap::PixelsResource pixelsResource(bitmap->GetPixelsResourceHandle());
            UInt8* pixels = pixelsResource.GetMutableData();
            if (pixels == 0) {
                return false;
            }

            textRenderContext.SetPenColor(Color(1.F, 1.F, 1.F, 1.F));
            static_cast<void>(textRenderContext.SetBitmap(Bitmap::SharedPointer(bitmap)));

            LayoutingOptions layoutingOptions = textBrushProperties.GetLayoutingOptions();
            layoutingOptions.SetOffset(TextCoordinate(-textLeft, -textTop));

            MemoryPlatform::Set(pixels, 0, static_cast<SizeType>(bitmap->GetWidth() * bitmap->GetHeight()));
            bool rendered = textRenderer.Render(
                textRenderContext,
                layoutingOptions,
                textBrushProperties.GetShapingOptions(),
                textBrushProperties.GetTextProperties());

            if (!rendered) {
                return false;
            }

            //upload the new bitmap
            return m_bitmapImage->Upload();
        }

        /******************************************************************************
         *  Render
         ******************************************************************************/
        void BitmapTextBrushCache::Render(const TextBrushProperties& /*textBrushProperties*/, const Matrix3x2& transform, const Rectangle& bound, ContextHandle2D output, Rectangle& outputArea)
        {
            if ((!m_bitmapImage.PointsToNull()) && m_bitmapImage->IsUploaded()) {
                Matrix3x2 correctedTransform(transform);
                correctedTransform.Translate(bound.GetLeft(), bound.GetTop());
                bool result = Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, correctedTransform);

                result = result && m_bitmapImage->Activate(output);
                result = result && RenderDevice2D::SetActiveArea(
                    output,
                    RenderDevice2D::SourceSurface,
                    0.0F,
                    0.0F,
                    bound.GetWidth(),
                    bound.GetHeight());
                result = result && Renderer2D::Blit(output);

                if (result) {
                    Float left;
                    Float top;
                    Float width;
                    Float height;
                    if (RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height)) {
                        outputArea = Rectangle(left, top, width, height);
                    }
                }
            }
        }
    }   // namespace Internal
}   // namespace Candera
