//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextBrush.h"

#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/TextEngine/Internal/StyleTools.h>

#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/BitmapTextBrushCache.h>
#ifdef CANDERA_2D_OVER_3D_ENABLED
#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/GlyphAtlasTextBrushCache.h>
#endif
#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/GlyphTextBrushCache.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/NoTextBrushCache.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/SurfaceTextBrushCache.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/SurfaceTextRenderContext.h>


namespace Candera {
    FEATSTD_RTTI_DEFINITION(TextBrush, BrushEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    TextBrush::TextBrush() :
        Base(),
        m_isUploaded(false),
        m_isMeasurementCacheValid(false),
        m_isRenderCacheValid(false),
        m_cache(0),
        m_properties()
    {
    }

    TextBrush::TextBrush(const TextBrush& rhs) :
        Base(rhs),
        m_isUploaded(false),
        m_isMeasurementCacheValid(false),
        m_isRenderCacheValid(false),
        m_cache(0),
        m_properties(rhs.m_properties)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    TextBrush::~TextBrush()
    {
        static_cast<void>(TextBrush::Unload());
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    TextBrush::SharedPointer TextBrush::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        TextBrush* brush = FEATSTD_NEW(TextBrush);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        TextBrush::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void TextBrush::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform, const Node2D& node,
                           ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "TextBrush"));

        if ((m_cache != 0) && (m_properties.m_text != 0)) {
            ClearTextRectangle(transform, output, outputArea);
            // After clearing outputArea is set to the maximum bound of the text.

            ActivateGlyphBlend(output);
            ActivateColor(node, output);

            Rectangle renderArea;
            RenderText(transform, output, renderArea);

            // renderArea might be smaller then outputArea for some caching configurations.
            outputArea.Intersect(renderArea);

            DeactivateColor(output);
            DeactivateGlyphBlend(output);
        }
    }

    /******************************************************************************
     *  GetBoundingRectangle
     ******************************************************************************/
    void TextBrush::GetBoundingRectangle(Rectangle& boundingRectangle) const
    {
        if (UpdateMeasurementCache()) {
            boundingRectangle = m_boundingRect;
        }
        else {
            boundingRectangle = Rectangle();
        }
    }

    /******************************************************************************
     *  GetLayoutingRectangle
     ******************************************************************************/
    void TextBrush::GetLayoutingRectangle(Rectangle& rectangle) const
    {
        rectangle.SetPosition(Vector2());
        if (UpdateMeasurementCache()) {
            rectangle.SetSize(m_layoutingRect.GetPosition() + m_layoutingRect.GetSize());
            if (m_properties.m_layoutingArea().GetX() > 0.F) {
                rectangle.SetWidth(m_properties.m_layoutingArea().GetX());
            }
            if (m_properties.m_layoutingArea().GetY() > 0.F) {
                rectangle.SetHeight(m_properties.m_layoutingArea().GetY());
            }
        }
        else {
            rectangle.SetSize(Vector2());
        }
    }

    /******************************************************************************
     *  GetBoundingRectangle
     ******************************************************************************/
    void TextBrush::GetBasePoint(Vector2& basePoint) const
    {
        if (UpdateMeasurementCache()) {
            basePoint.SetX((m_layoutingRect.GetLeft() + m_layoutingRect.GetWidth()) * 0.5F);
            const TextRendering::Metrics& metrics = m_properties.m_style()->GetMetrics();
            Float ascender = static_cast<Float>(metrics.ascender -  1);
            basePoint.SetY(m_layoutingRect.GetTop() + ascender);
        }
        else {
            basePoint.SetZero();
        }
    }

    /******************************************************************************
     *  ActivateColor
     ******************************************************************************/
    void TextBrush::ActivateColor(const Node2D& node, ContextHandle2D output)
    {
        const Color::Data& color = Color().Get();
        Float cameraAlpha = 1.0F;
        const Camera2D* camera2D = Renderer2D::GetActiveCamera();
        if (0 != camera2D) {
            if (camera2D->IsCameraEffectiveAlphaEnabled()) {
                cameraAlpha = camera2D->GetEffectiveAlphaValue();
            }
        }
        static_cast<void>(RenderDevice2D::SetSurfaceConstColor(
            output,
            RenderDevice2D::SourceSurface,
            color.red,
            color.green,
            color.blue,
            color.alpha * node.GetEffectiveAlphaValue() * cameraAlpha));
    }

    /******************************************************************************
     *  DeactivateColor
     ******************************************************************************/
    void TextBrush::DeactivateColor(ContextHandle2D output) const
    {
        static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.F, 1.F, 1.F, 1.F));
    }

    /******************************************************************************
     *  ActivateGlyphBlend
     ******************************************************************************/
    void TextBrush::ActivateGlyphBlend(ContextHandle2D output)
    {
        if ((m_cache != 0) && (m_cache->IsGlyphByGlyph())) {
            static_cast<void>(RenderDevice2D::SetBlendOperation(
                output,
                RenderDevice2D::SourceSurface,
                RenderDevice2D::Add,
                RenderDevice2D::One,
                RenderDevice2D::One));
        }
    }

    /******************************************************************************
     *  DeactivateGlyphBlend
     ******************************************************************************/
    void TextBrush::DeactivateGlyphBlend(ContextHandle2D output)
    {
        if ((m_cache != 0) && (m_cache->IsGlyphByGlyph())) {
            static_cast<void>(RenderDevice2D::SetBlendOperation(
                output,
                RenderDevice2D::SourceSurface,
                RenderDevice2D::Add,
                RenderDevice2D::One,
                RenderDevice2D::Zero));
        }
    }

    /******************************************************************************
     *  RenderText
     ******************************************************************************/
    void TextBrush::RenderText(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea)
    {
        if (m_cache != 0) {
            static_cast<void>(RenderDevice2D::SetActiveArea(output, RenderDevice2D::SourceSurface, 0.0F, 0.0F, -1.0F, -1.0F));
            static_cast<void>(UpdateMeasurementCache());
            m_cache->Render(m_properties, transform, m_boundingRect, output, outputArea);
        }
    }

    /******************************************************************************
     *  ClearTextRectangle
     ******************************************************************************/
    void TextBrush::ClearTextRectangle(const Matrix3x2& transform, ContextHandle2D output, Rectangle& outputArea)
    {
        if ((m_cache != 0) && (m_properties.m_text() != 0)) {
            //rectangle with corrected base line required.
            //use GetBoundingRectangle instead of m_boundingRect.
            GetBoundingRectangle(outputArea);
            outputArea.Transform(transform);

            if (m_cache->IsGlyphByGlyph()) {
                bool result = RenderDevice2D::SetBlendOperation(
                    output,
                    RenderDevice2D::SourceSurface,
                    RenderDevice2D::Add,
                    RenderDevice2D::One,
                    RenderDevice2D::Zero);

                result = result && RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 0.0F, 0.0F, 0.0F, 0.0F);
                result = result && RenderDevice2D::SetActiveArea(
                    output,
                    RenderDevice2D::SourceSurface,
                    outputArea.GetLeft(),
                    outputArea.GetTop(),
                    outputArea.GetWidth(),
                    outputArea.GetHeight());
                result = result && Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, Matrix3x2());
                result = result && RenderDevice2D::SetSurface(output, RenderDevice2D::SourceSurface, 0);

                if (result) {
                    static_cast<void>(Renderer2D::Blit(output));
                }
            }
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool TextBrush::Upload()
    {
        if (m_isUploaded != 0) {
            return false;
        }

        m_properties.m_cacheType.SetChanged(true);

        m_isUploaded = true;
        m_isUploaded = Update();

        if (m_isUploaded && (m_cache != 0))
        {
            m_cache->Upload();
        }
        return m_isUploaded;
    }
    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool TextBrush::Unload()
    {
        if (m_isUploaded == 0) {
            return false;
        }

        if (m_cache != 0) {
            m_cache->Unload();
            if (!m_cache->IsGlyphByGlyph()) {
                FEATSTD_DELETE(m_cache);
            }
            m_cache = 0;
        }

        m_isUploaded = false;
        return true;
    }
    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool TextBrush::Update()
    {
        if (m_isUploaded == 0) {
            return false;
        }

        if (m_properties.m_cacheType.IsChanged()) {
            SetRenderCacheValid(false);
            SetMeasurementCacheValid(false);

            if ((m_cache != 0) && (!m_cache->IsGlyphByGlyph())) {
                FEATSTD_DELETE(m_cache);
            }

            static Internal::NoTextBrushCache s_cacheNoCache;
            static Internal::GlyphTextBrushCache s_cacheGlyphCache;

            switch(m_properties.m_cacheType()) {
                case NoCache:
                    m_cache = &s_cacheNoCache;
                    break;

                case GlyphCache:
                    m_cache = &s_cacheGlyphCache;
                    break;

                case BitmapCache:
                    m_cache = FEATSTD_NEW(Internal::BitmapTextBrushCache);
                    break;

                case SurfaceCache:
                    m_cache = FEATSTD_NEW(Internal::SurfaceTextBrushCache);
                    break;

#ifdef CANDERA_2D_OVER_3D_ENABLED
                case GlyphAtlasCache:
                    m_cache = FEATSTD_NEW(Internal::GlyphAtlasTextBrushCache);
                    break;
#endif

                default:
                    m_cache = 0;
            }

            if (m_cache != 0) {
                m_properties.m_cacheType.SetChanged(false);
            }
        }

        if (m_cache == 0) {
            return false;
        }

        if (!IsRenderCacheValid()){
            static_cast<void>(UpdateMeasurementCache());

            if (!m_cache->Update(m_properties, m_boundingRect)) {
                if (!m_cache->IsGlyphByGlyph()) {
                    FEATSTD_DELETE(m_cache);
                }
                m_cache = 0;
                return false;
            }
            SetRenderCacheValid(true);
        }

        return true;
    }

    /******************************************************************************
     *  TextRectToRectangle
     ******************************************************************************/
    inline void TextRectToRectangle(Rectangle& out, const TextRendering::TextRect& in)
    {
        out.SetTop(static_cast<Float>(in.GetTop()));
        out.SetLeft(static_cast<Float>(in.GetLeft()));
        out.SetWidth(static_cast<Float>(in.GetWidth()));
        out.SetHeight(static_cast<Float>(in.GetHeight()));
    }

    /******************************************************************************
     *  UpdatedBoundingRectangle
     ******************************************************************************/
    bool TextBrush::UpdateMeasurementCache() const
    {
        if (IsMeasurementCacheValid()) {
            return true;
        }

        if ((m_properties.m_text.Get() == 0) || (m_properties.m_style().PointsToNull())) {
            return false;
        }

        TextRendering::CursorTextMeasureContext cursorTextMeasureContext(*m_properties.m_style(),
                (m_cache != 0) ? m_cache->GetTextRenderContext() : 0);
        TextRendering::GlyphTextMeasureContext glyphTextMeasureContext (
                (m_cache != 0) ? m_cache->GetTextRenderContext() : 0);

        glyphTextMeasureContext.SetNextContext(&cursorTextMeasureContext);

        static_cast<void>(TextRendering::TextRenderer().Render(
            glyphTextMeasureContext,
            m_properties.GetLayoutingOptions(),
            m_properties.GetShapingOptions(),
            m_properties.GetTextProperties()));

        TextRectToRectangle(m_boundingRect, glyphTextMeasureContext.GetTextRectangle());
        TextRectToRectangle(m_layoutingRect, cursorTextMeasureContext.GetTextRectangle());

        // Correct bounding rectangle when caching is enabled.
        const bool isGlyphByGlyphCache =
            (m_properties.m_cacheType == NoCache) ||
            (m_properties.m_cacheType == GlyphCache);
        if (!isGlyphByGlyphCache) {
            Vector2 cacheSize = m_properties.m_cacheArea();
            if (cacheSize.GetX() > 0.F) {
                if (cacheSize.GetX() < m_boundingRect.GetWidth()) {
                    m_boundingRect.SetWidth(cacheSize.GetX());
                }
            }
            if (cacheSize.GetY() > 0.F) {
                if (cacheSize.GetY() < m_boundingRect.GetHeight()) {
                    m_boundingRect.SetHeight(cacheSize.GetY());
                }
            }
        }

        SetMeasurementCacheValid(true);

        return true;
    }

    /******************************************************************************
     *  Cached values validity
     ******************************************************************************/
    bool TextBrush::IsMeasurementCacheValid() const
    {
        UpdateCacheValidity();
        return m_isMeasurementCacheValid;
    }
    void TextBrush::SetMeasurementCacheValid(bool valid) const
    {
        m_isMeasurementCacheValid = valid;
    }
    bool TextBrush::IsRenderCacheValid() const
    {
        UpdateCacheValidity();
        return m_isRenderCacheValid;
    }
    void TextBrush::SetRenderCacheValid(bool valid) const
    {
        m_isRenderCacheValid = valid;
    }

    void TextBrush::UpdateCacheValidity() const
    {
        if (!m_properties.m_style.Get().PointsToNull()) {
            UInt16 newStyleVersion = Candera::TextRendering::Internal::StyleTools::GetStyleVersion(*(m_properties.m_style.Get()));
            if (newStyleVersion != m_properties.m_currentStyleVersion.Get()) {
                const_cast<UInt16PropertyEx&>(m_properties.m_currentStyleVersion).Set(newStyleVersion);
            }
        }
        bool isChanged =
            m_properties.m_text.IsChanged() ||
            m_properties.m_style.IsChanged() ||
            m_properties.m_currentStyleVersion.IsChanged() ||
            m_properties.m_culture.IsChanged() ||
            m_properties.m_complexScriptMode.IsChanged() ||
            m_properties.m_layoutingArea.IsChanged() ||
            m_properties.m_cacheArea.IsChanged() ||
            m_properties.m_glyphSpacing.IsChanged() ||
            m_properties.m_lineSpacing.IsChanged() ||
            m_properties.m_pixelwiseLineSpacing.IsChanged() ||
            m_properties.m_horizontalAlignment.IsChanged() ||
            m_properties.m_verticalAlignment.IsChanged() ||
            m_properties.m_multiLineLayouting.IsChanged() ||
            m_properties.m_wordWrap.IsChanged() ||
            m_properties.m_layoutStrategy.IsChanged() ||
            m_properties.m_preprocessedText.IsChanged();
        if (isChanged) {
            m_isMeasurementCacheValid = false;
            m_isRenderCacheValid = false;

            const_cast<StylePropertyEx&>(m_properties.m_style).SetChanged(false);
            const_cast<UInt16PropertyEx&>(m_properties.m_currentStyleVersion).SetChanged(false);
            const_cast<CulturePropertyEx&>(m_properties.m_culture).SetChanged(false);
            const_cast<TextPropertyEx&>(m_properties.m_text).SetChanged(false);
            const_cast<TextBrushComplexScriptModePropertyEx&>(m_properties.m_complexScriptMode).SetChanged(false);
            const_cast<Vector2PropertyEx&>(m_properties.m_layoutingArea).SetChanged(false);
            const_cast<Vector2PropertyEx&>(m_properties.m_cacheArea).SetChanged(false);
            const_cast<FloatPropertyEx&>(m_properties.m_glyphSpacing).SetChanged(false);
            const_cast<FloatPropertyEx&>(m_properties.m_lineSpacing).SetChanged(false);
            const_cast<BoolPropertyEx&>(m_properties.m_pixelwiseLineSpacing).SetChanged(false);
            const_cast<HorizontalAlignmentPropertyEx&>(m_properties.m_horizontalAlignment).SetChanged(false);
            const_cast<VerticalAlignmentPropertyEx&>(m_properties.m_verticalAlignment).SetChanged(false);
            const_cast<BoolPropertyEx&>(m_properties.m_multiLineLayouting).SetChanged(false);
            const_cast<BoolPropertyEx&>(m_properties.m_wordWrap).SetChanged(false);
            const_cast<TextLayoutStrategyPropertyEx&>(m_properties.m_layoutStrategy).SetChanged(false);
            const_cast<PreprocessedTextPropertyEx&>(m_properties.m_preprocessedText).SetChanged(false);
        }
    }


    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer TextBrush::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(TextBrush)(*this));
    }

}   // namespace Candera
