//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SOLID_COLOR_BRUSH_BLEND_H)
    #define CANDERA_SOLID_COLOR_BRUSH_BLEND_H

#include <Candera/Engine2D/Effects/CombinedEffect2D.h>
#include <CanderaPlatform/Device/Common/Effects/SolidColorBrush.h>
#include <CanderaPlatform/Device/Common/Effects/BlendEffect.h>

namespace Candera {

    class Image2D;

    /**
     *  @brief  Output rectangle filled with a solid color and blend it with the store buffer.
     *          Same as chaining (SolidColorBrush + BlendEffect)
     *          @see Candera::SolidColorBrush
     *          @see Candera::BlendEffect
     */
    class SolidColorBrushBlend : public CombinedEffect2D {

        FEATSTD_TYPEDEF_BASE(CombinedEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<SolidColorBrushBlend> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~SolidColorBrushBlend();

        // overrides Effect2D::GetBrushEffect2D
        virtual BrushEffect2D* GetBrushEffect2D() { return &m_solidBrush; }

        // overrides Effect2D::GetBlendEffect2D
        virtual BlendEffect2D* GetBlendEffect2D() { return &m_blendEffect; }

        /**
         *  Retrieves the solid color brush.
         *  @return     The solid color brush.
         */
        const SolidColorBrush& GetSolidColorBrush() const { return m_solidBrush; }
        SolidColorBrush& GetSolidColorBrush() { return m_solidBrush; }

        /**
         *  Retrieves the blend effect.
         *  @return     The blend effect.
         */
        const BlendEffect& GetBlendEffect() const { return m_blendEffect; }
        BlendEffect& GetBlendEffect() { return m_blendEffect; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /**
        * Mesh2D cannot be supported by all effects, as certain effects would create 2D vertex buffers, which would be overriden by the Mesh2D.
        * This flag helps to distinguish between effects that can be used with Mesh2D and effects that are not available.
        * @return Whether Mesh2D can be used with this effect (true) or not (false).
        */
        virtual bool IsMesh2DSupported() const override { return true; }

        /// @cond excluded from doxygen
        CdaEffect2DCombinedDef(Candera::SolidColorBrushBlend, SolidColorBrushBlend,  EFFECT2D_TYPE_BRUSH, EFFECT2D_TYPE_BLEND)
            CdaEffect2DProperties()

                CdaEffect2DImportProperties(SolidColorBrush, Candera::SolidColorBrush, m_solidBrush)
                CdaEffect2DImportProperties(BlendEffect, Candera::BlendEffect, m_blendEffect)

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        SolidColorBrushBlend();
        explicit SolidColorBrushBlend(const SolidColorBrushBlend& rhs);
        SolidColorBrushBlend& operator = (const SolidColorBrushBlend& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        // Use class wrapper to forward protected members.
        class SolidColorBrushWrapper : public SolidColorBrush {} m_solidBrush;
        class BlendEffectWrapper : public BlendEffect {} m_blendEffect;
    };

}   // namespace Candera

#endif  // CANDERA_SOLID_COLOR_BRUSH_BLEND_H
