//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ShadowEffect.h"
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Mathematics/Matrix4.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(ShadowEffect, InPlaceEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    ShadowEffect::ShadowEffect() :
        m_shadowColor(Color(1.0F, 1.0F, 1.0F, 1.0F)),
        m_shadowScale(Vector2(1.0F, 1.0F)),
        m_shadowPositionOffset(Vector2(0.0F, 0.0F))
    {
    }

    ShadowEffect::ShadowEffect(const ShadowEffect& rhs) :
        Base(rhs),
        m_shadowColor(rhs.m_shadowColor.Get()),
        m_shadowScale(rhs.m_shadowScale.Get()),
        m_shadowPositionOffset(rhs.m_shadowPositionOffset.Get())
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    ShadowEffect::~ShadowEffect()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    ShadowEffect::SharedPointer ShadowEffect::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        ShadowEffect* brush = FEATSTD_NEW(ShadowEffect);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        ShadowEffect::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void ShadowEffect::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node,
                              ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "ShadowEffect"));

        static_cast<void>(RenderDevice2D::SetSurface(output, RenderDevice2D::SourceSurface, input));

        Matrix3x2 shadowTransformation(transform);

        //transform shadow
        Float deltaX = ((inputArea.GetWidth() * m_shadowScale().GetX()) - inputArea.GetWidth()) * 0.5F;
        Float deltaY = ((inputArea.GetHeight() * m_shadowScale().GetY()) - inputArea.GetHeight()) * 0.5F;
        shadowTransformation.Translate(inputArea.GetLeft() - deltaX, inputArea.GetTop()  - deltaY);
        shadowTransformation.Scale(m_shadowScale().GetX(), m_shadowScale().GetY());
        Matrix3x2 translation;
        translation.SetTranslation(inputArea.GetLeft() + m_shadowPositionOffset().GetX(), inputArea.GetTop() + m_shadowPositionOffset().GetY());
        shadowTransformation = shadowTransformation * translation;
        static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, shadowTransformation));

        //set matrix to discard colors and take set color instead, preserve alpha
        Matrix4 colorSetMatrix;
        colorSetMatrix.SetZero();
        colorSetMatrix.Set(3, 3, 1.0F);
        colorSetMatrix.Set(3, 0, m_shadowColor().red);
        colorSetMatrix.Set(3, 1, m_shadowColor().green);
        colorSetMatrix.Set(3, 2, m_shadowColor().blue);
        static_cast<void>(RenderDevice2D::SetColorTransformation(output, RenderDevice2D::SourceSurface, colorSetMatrix));

        static_cast<void>(RenderDevice2D::SetActiveArea(output, RenderDevice2D::SourceSurface, 0.0F, 0.0F, -1.0F, -1.0F));
        //set alpha channel
        Float cameraAlpha = 1.0F;
        const Camera2D* camera2D = Renderer2D::GetActiveCamera();
        if (0 != camera2D) {
            if (camera2D->IsCameraEffectiveAlphaEnabled()) {
                cameraAlpha = camera2D->GetEffectiveAlphaValue();
            }
        }
        static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.0F, 1.0F, 1.0F, m_shadowColor().alpha * node.GetEffectiveAlphaValue() * cameraAlpha));
        //blit shadow
        static_cast<void>(Renderer2D::Blit(output));
        //unset alpha channel
        static_cast<void>(RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.0F, 1.0F, 1.0F, 1.0F));

        //deactivate color transformation
        colorSetMatrix.SetIdentity();
        static_cast<void>(RenderDevice2D::SetColorTransformation(output, RenderDevice2D::SourceSurface, colorSetMatrix));

        outputArea = inputArea;

        //draw the actual brush
        static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));
        static_cast<void>(Renderer2D::Blit(output));
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer ShadowEffect::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(ShadowEffect)(*this));
    }

}   // namespace Candera
