//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MIRROR_EFFECT_H)
    #define CANDERA_MIRROR_EFFECT_H

#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Vector2Property.h>
#include <Candera/Engine2D/Property/Property.h>
#include <Candera/System/Mathematics/MathDataTypes.h>
#include <CanderaPlatform/Device/Common/Effects/EffectDataTypes.h>

namespace Candera {
    /**
     *  @brief  Applies a mirror reflection to the predecessor effect,
     *          Alpha gradient can be turned on or off.
     */
    class MirrorEffect : public InPlaceEffect2D {
        FEATSTD_TYPEDEF_BASE(InPlaceEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<MirrorEffect> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class.
         *  @return Pointer to the created MirrorEffect object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~MirrorEffect();

        /**
         *  Retrieves the MirrorAxisFrom Vector of the MirrorEffect as a Vector2Property,
         *  this returned property can be changed. The performed changes will affect this class.
         *  @return The From-Vector of the mirror axis.
         */
        Vector2Property& MirrorAxisFrom() { return m_mirrorAxisFrom; }

        /**
        *  Retrieves the MirrorAxisTo Vector of the MirrorEffect as a Vector2Property,
        *  this returned property can be changed. The performed changes will affect this class.
         *  @return The To-Vector of the mirror axis.
         */
        Vector2Property& MirrorAxisTo() { return m_mirrorAxisTo; }

        /**
         *  Retrieves the alpha value of the MirrorEffect, this returned property
         *  can be changed. The performed changes will affect this class.
         *  @return The current alpha value of the MirrorEffect.
         */
        FloatProperty& Alpha() { return m_alpha; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::MirrorEffect, MirrorEffect, EFFECT2D_TYPE_INPLACE)
            CdaEffect2DProperties()

                CdaEffect2DProperty(MirrorAxisFrom, Vector2, m_mirrorAxisFrom)
                    CdaDescription("First point on axis to mirror about. ")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(MirrorAxisTo, Vector2, m_mirrorAxisTo)
                    CdaDescription("Second point on axis to mirror about. ")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(Alpha, Float, m_alpha)
                    CdaDescription("Alpha value to apply to mirrored image.")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        MirrorEffect();
        MirrorEffect(const MirrorEffect& rhs);
        MirrorEffect& operator = (const MirrorEffect& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        Vector2Property m_mirrorAxisFrom;
        Vector2Property m_mirrorAxisTo;
        FloatProperty m_alpha;
        Vector2 m_lastMirrorAxisFrom;
        Vector2 m_lastMirrorAxisTo;

        Matrix3x2 m_mirrorMatrix;

        bool IsMirrorMatrixValid();
        void UpdateMirrorMatrix();
    };
}   // namespace Candera

#endif  // CANDERA_MIRROR_EFFECT_H
