//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MASK_EFFECT_H)
    #define CANDERA_MASK_EFFECT_H

#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Property.h>
#include <Candera/Engine2D/Property/ColorProperty.h>
#include <Candera/Engine2D/Property/Image2DProperty.h>
#include <Candera/Engine2D/Property/Node2DProperty.h>
#include <CanderaPlatform/Device/Common/Effects/EffectDataTypes.h>
#include <CanderaPlatform/Device/Common/Effects/MaskProperties.h>

namespace Candera {

    /**
     *  @brief Apply an alpha mask to an image.
     *
     *  There are no other applied transformations.
     *  The position of the mask is controlled through a node, that can be either
     *  the current node, or some other specified node. The mask can't be rotated or scaled.
     *  Only the alpha channel of the mask is used to either replace or modulate the alpha
     *  channel of the source image.
     */
    class MaskEffect : public InPlaceEffect2D {

        FEATSTD_TYPEDEF_BASE(InPlaceEffect2D);

    public:

        typedef MemoryManagement::SharedPointer<MaskEffect> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        static SharedPointer Create();

        /**
         *  Destructor
         */
        virtual ~MaskEffect();

        /**
         *  Activate the effect on a 2D context.
         *  @param node   Node 2d
         *  @param output The effect is activated on this 2D context.
         */
        void ActivateMask(const Node2D& node, ContextHandle2D output);

        /**
         *  Deactivate the effect.
         *  @param output The effect is deactivated for this 2D context.
         */
        void DeactivateMask(ContextHandle2D output) const;

        // overrides Effect2D::Upload
        virtual bool Upload();

        // overrides Effect2D::Unload
        virtual bool Unload();

        // overrides Effect2D::Update
        virtual bool Update();

        // overrides Effect2D::IsUploaded
        virtual bool IsUploaded() const { return m_uploaded; }

        /**
         *  Retrieves the current mask node. This returned property
         *  can be changed. The performed changes will affect this class.
         *  Property used as position of mask on screen. If set to 0 the
         *  position of the current node is used.
         *  The mask ignores the scale, the rotation and the pivot of the node.
         *  @return     The mask node property.
         */
        Node2DProperty& MaskNode() { return m_maskNode; }

        /**
         *  Retrieves the alpha value used outside the mask. This returned property
         *  can be changed. The performed changes will affect this class.
         *  Property used as alpha value outside the mask image.
         *  @return     The alpha value used outside the mask.
         */
        FloatProperty& OutsideMaskAlphaValue() { return m_outsideMaskAlphaValue; }

        /**
         *  Retrieves the property of the current mask blend mode. This returned property
         *  can be changed. The performed changes will affect this class.
         *  Property used to choose the blending mode used between the mask and the source image.
         *  Blending can be either Replace, in which the source alpha channel is simply replaced
         *  by the destination alpha channel, or Add, in which case the source alpha channel is modulated
         *  by the mask alpha channel.
         *  @return     The mask blend mode property.
         */
        MaskBlendModeProperty& MaskBlendMode() { return m_maskBlendMode; }

        /**
         *  Retrieves the current mask. This returned property
         *  can be changed. The performed changes will affect this class.
         *  Property used as mask image. Only the alpha channel of the mask is used.
         *  @return     The currently set mask image.
         */
        Image2DProperty& Mask() { return m_mask; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::MaskEffect, MaskEffect, EFFECT2D_TYPE_INPLACE)
            CdaEffect2DProperties()

                CdaEffect2DProperty(MaskNode, Node2D*, m_maskNode)
                    CdaDescription("Node used to control the position of the mask. "
                        "If set to 0 the position of the current node is used. "
                        "The mask ignores the scale, the rotation and the pivot of the node.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(OutsideMaskAlphaValue, Float, m_outsideMaskAlphaValue)
                    CdaDescription("The alpha value used as mask value outside the mask image.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(MaskBlendMode, MaskBlendModeType, m_maskBlendMode)
                    CdaDescription("The blend operation used when combining mask alpha "
                        "with source alpha. The replace operation replaces the alpha channel "
                        "of the source. It shouldn't be used with alpha only source images, "
                        "like text. The And-operation modulates the alpha channel of the source "
                        "by the alpha channel of the mask. The restrictions of this are that "
                        "the source and the mask shouldn't both have intermediate alpha values "
                        "between opaque and fully transparent (note this allows to have "
                        "alpha gradients in either the source image or the mask, but not in both), "
                        "and the mask should have an 8 bit alpha channel.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(Mask, MemoryManagement::SharedPointer<Image2D>, m_mask)
                    CdaDescription("The image used as mask. Only the alpha channel of the mask is used.")
                    CdaUpdateType(OnApplyUpdateType)
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        MaskEffect();
        explicit MaskEffect(const MaskEffect& rhs);
        MaskEffect& operator = (const MaskEffect& rhs);

        // overrides Effect2D::IsUploaded
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        bool m_uploaded;

        Node2DProperty m_maskNode;
        FloatProperty m_outsideMaskAlphaValue;
        MaskBlendModeProperty m_maskBlendMode;
        Image2DProperty m_mask;

        MemoryManagement::SharedPointer<Image2D> m_uploadedMask;
    };

}   // namespace Candera

#endif  // CANDERA_MASK_EFFECT_H
