//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_HSL_CORRECTION_EFFECT_H)
    #define CANDERA_HSL_CORRECTION_EFFECT_H

#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/Property.h>

namespace Candera {

    class Image2D;
    class Matrix4;

    /**
     *  @brief Apply hue, saturation and lightness correction to an image.
     */
    class HslCorrectionEffect : public InPlaceEffect2D {

        FEATSTD_TYPEDEF_BASE(InPlaceEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<HslCorrectionEffect> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~HslCorrectionEffect();

        /**
         *  Retrieves current the hue value, this returned property
         *  can be changed. The performed changes will affect this class.
         *  Value used to rotate the color channels. Range -180..180.
         *  The value is periodical outside range.
         *  @return     The hue property.
         */
        FloatProperty& Hue() { return m_hue; }

        /**
         *  Retrieves the current saturation value, this returned property
         *  can be changed. The performed changes will affect this class.
         *  Value used to saturate color. 0 means grayscale, 1 means unchanged,
         *  -1 means inverted.
         *  @return     The saturation property.
         */
        FloatProperty& Saturation() { return m_saturation; }

        /**
         *  Retrieves the current lightness value, this returned property
         *  can be changed. The performed changes will affect this class.
         *  Value used to change brightness. 0 means black, 1 means unchanged.
         *  @return     The lightness property.
         */
        FloatProperty& Lightness() { return m_lightness; }

        /**
         *  Activate the effect on a 2D context.
         *  @param output The effect is activated for this 2d context.
         */
        void ActivateHslCorrection(ContextHandle2D output) const;

        /**
         *  Deactivate the effect.
         *  @param output The effect is deativated for this 2d context.
         */
        void DeactivateHslCorrection(ContextHandle2D output) const;

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /// @cond excluded from doxygen
        CdaEffect2DDef(Candera::HslCorrectionEffect, HslCorrectionEffect, EFFECT2D_TYPE_INPLACE)
            CdaEffect2DProperties()

                CdaEffect2DProperty(Hue, Float, m_hue)
                    CdaDescription("Value used to rotate the color channels. Range -180..180. "
                                   "The value is periodical outside range.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(Saturation, Float, m_saturation)
                    CdaDescription("Value used to saturate color. 0 means grayscale, 1 means unchanged, "
                                    "-1 means inverted.")
                CdaEffect2DPropertyEnd()

                CdaEffect2DProperty(Lightness, Float, m_lightness)
                    CdaDescription("Value used to change brighness. 0 means black, 1 means unchanged.")
                CdaEffect2DPropertyEnd()

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

        /**
         *  Compute a color transformation matrix based on hue, saturation and lightness correction parameters.
         *  @param hue Hue.
         *  @param saturation Saturation.
         *  @param lightness Lightness.
         *  @param output Output matrix.
         */
        static void ComputeMatrix(Float hue, Float saturation, Float lightness, Matrix4& output);

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        HslCorrectionEffect();
        explicit HslCorrectionEffect(const HslCorrectionEffect& rhs);
        HslCorrectionEffect& operator = (const HslCorrectionEffect& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        FloatProperty m_hue;
        FloatProperty m_saturation;
        FloatProperty m_lightness;
    };

}   // namespace Candera

#endif  // CANDERA_HSL_CORRECTION_EFFECT_H
