//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlSolidColorBrushMaskBlend.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(GlSolidColorBrushMaskBlend, CombinedEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
     GlSolidColorBrushMaskBlend::GlSolidColorBrushMaskBlend()
    {
    }

    GlSolidColorBrushMaskBlend::GlSolidColorBrushMaskBlend(const GlSolidColorBrushMaskBlend& rhs) :
        Base(rhs),
        m_solidBrush(rhs.m_solidBrush),
        m_maskEffect(rhs.m_maskEffect),
        m_blendEffect(rhs.m_blendEffect)
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlSolidColorBrushMaskBlend::~GlSolidColorBrushMaskBlend()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlSolidColorBrushMaskBlend::SharedPointer GlSolidColorBrushMaskBlend::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        GlSolidColorBrushMaskBlend* brush = FEATSTD_NEW(GlSolidColorBrushMaskBlend);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlSolidColorBrushMaskBlend::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlSolidColorBrushMaskBlend::Render(SurfaceHandle /*input*/, const Rectangle& /*inputArea*/, const Matrix3x2& transform,
                                               const Node2D& node, ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "SolidColorBrushMaskBlend"));

        const Vector2& size = m_solidBrush.Size();

        if ((size.GetX() != 0.0F) && (size.GetY() != 0.0F)) {
            m_blendEffect.ActivateBlend(output);

            m_maskEffect.ActivateMask(node, output);

            m_solidBrush.ActivateColor(node, output);
            m_solidBrush.RenderSolidRectangle(transform, output, outputArea, node);
            m_solidBrush.DeactivateColor(output);

            m_maskEffect.DeactivateMask(output);

            m_blendEffect.DeactivateBlend(output);
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool GlSolidColorBrushMaskBlend::Upload()
    {
        bool success = m_solidBrush.Upload();
        success = m_maskEffect.Upload() && success;
        success = m_blendEffect.Upload() && success;
        return success;
    }
    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool GlSolidColorBrushMaskBlend::Unload()
    {
        bool success = m_solidBrush.Unload();
        success = m_maskEffect.Unload() && success;
        success = m_blendEffect.Unload() && success;
        return success;
    }
    /******************************************************************************
     *  Update
     ******************************************************************************/
    bool GlSolidColorBrushMaskBlend::Update()
    {
        bool success = m_solidBrush.Update();
        success = m_maskEffect.Update() && success;
        success = m_blendEffect.Update() && success;
        return success;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer GlSolidColorBrushMaskBlend::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlSolidColorBrushMaskBlend)(*this));
    }

}   // namespace Candera
