//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "GlMaskEffect.h"
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>

namespace Candera {
    FEATSTD_RTTI_DEFINITION(GlMaskEffect, InPlaceEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    GlMaskEffect::GlMaskEffect() :
        m_uploaded(false)
    {
    }

    GlMaskEffect::GlMaskEffect(const GlMaskEffect& rhs) :
        Base(rhs),
        m_uploaded(false),
        m_maskNode(rhs.m_maskNode.Get()),
        m_mask(const_cast<Image2D*>(rhs.m_mask.Get()))
    {
         if (rhs.IsUploaded()) {
            static_cast<void>(GlMaskEffect::Upload());
        }
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    GlMaskEffect::~GlMaskEffect()
    {
        static_cast<void>(GlMaskEffect::Unload());
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    GlMaskEffect::SharedPointer GlMaskEffect::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        GlMaskEffect* brush = FEATSTD_NEW(GlMaskEffect);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        GlMaskEffect::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  ActivateMask
     ******************************************************************************/
    void GlMaskEffect::ActivateMask(const Node2D& node, ContextHandle2D output)
    {
        if (!m_uploadedMask.PointsToNull()) {
            Camera2D* cam = Renderer2D::GetActiveCamera();
            Matrix3x2 transform;
            bool snap_to_pixel = false;
            if (m_maskNode != 0) {
                transform = m_maskNode()->GetWorldTransform();

                RenderNode* mask_rendernode = Dynamic_Cast<RenderNode*, Node2D*>(m_maskNode());
                if (mask_rendernode != 0) {
                    snap_to_pixel = mask_rendernode->IsSnapToDevicePixelEnabled();
                }
            }
            else {
                transform = node.GetWorldTransform();
                const RenderNode* rendernode = Dynamic_Cast<const RenderNode*, const Node2D*>(&node);
                if ( rendernode != 0 ) {
                    snap_to_pixel = rendernode->IsSnapToDevicePixelEnabled();
                }
            }

#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
            const Vector2& perTranslate = Renderer2D::GetPreTranslate();
            transform.PreTranslate(perTranslate.GetX(), perTranslate.GetY());
#endif

            if (cam != 0) {
                Matrix3x2 viewMatrix = cam->GetViewMatrix();
                const Vector2& cameraTranslationOffset = Renderer2D::GetCameraTranslationOffset();
                viewMatrix.Translate(cameraTranslationOffset.GetX(), cameraTranslationOffset.GetY());
                transform *= viewMatrix;
            }
            if ( snap_to_pixel ) {
                Int32 x = Int32(0.5F + transform.Get(2, 0));
                transform.Set(2, 0, Float(x));
                Int32 y = Int32(0.5F + transform.Get(2, 1));
                transform.Set(2, 1, Float(y));
            }

            bool result = Renderer2D::SetTransformationMatrix(
                output,
                RenderDevice2D::MaskSurface,
                transform);

            result = result && RenderDevice2D::SetSurfaceConstColor(
                output,
                RenderDevice2D::MaskSurface,
                1.F,
                1.F,
                1.F,
                1.F);

            if (result) {
                static_cast<void>(m_uploadedMask->Activate(output, RenderDevice2D::MaskSurface));
            }
        }
    }
    /******************************************************************************
     *  DeactivateMask
     ******************************************************************************/
    void GlMaskEffect::DeactivateMask(ContextHandle2D output) const
    {
        static_cast<void>(RenderDevice2D::SetSurface(output,
                       RenderDevice2D::MaskSurface,
                       0));
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void GlMaskEffect::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node,
                            ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (Candera::PerfMon::TimingRecId::RenderEffect2D, "GlMaskEffect"));

        if (!m_uploadedMask.PointsToNull()) {
            static_cast<void>(Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform));
            static_cast<void>(RenderDevice2D::SetSurface(output, RenderDevice2D::SourceSurface, input));

            static_cast<void>(RenderDevice2D::SetActiveArea(output,
                              RenderDevice2D::SourceSurface,
                              inputArea.GetLeft(),
                              inputArea.GetTop(),
                              inputArea.GetWidth(),
                              inputArea.GetHeight()));

            ActivateMask(node, output);
            static_cast<void>(Renderer2D::Blit(output));
            DeactivateMask(output);

            Float left;
            Float top;
            Float width;
            Float height;
            if (RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height)) {
                outputArea = Rectangle(left, top, width, height);
            }
        }
    }

    /******************************************************************************
     *  Upload
     ******************************************************************************/
    bool GlMaskEffect::Upload()
    {
        if (m_uploaded) {
            return false;
        }

        if (m_uploadedMask.PointsToNull() && (m_mask() != 0)) {
            if (!m_mask()->Upload()){
                return false;
            }
            m_uploadedMask = MemoryManagement::SharedPointer<Image2D>(m_mask());
        }

        m_uploaded = true;
        return true;
    }

    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool GlMaskEffect::Unload()
    {
        if (!m_uploaded) {
            return false;
        }

        if (!m_uploadedMask.PointsToNull()) {
            if (!m_uploadedMask->Unload()) {
                return false;
            }
            m_uploadedMask.Release();
        }

        m_uploaded = false;

        return true;
    }

    /******************************************************************************
     *  Unload
     ******************************************************************************/
    bool GlMaskEffect::Update()
    {
        if (!m_uploaded) {
            return false;
        }
        bool success = true;
        if (m_mask != m_uploadedMask.GetPointerToSharedInstance()) {
            success =
                (m_uploadedMask.PointsToNull() || m_uploadedMask->Unload()) &&
                ((m_mask() == 0) || m_mask()->Upload());

            m_uploadedMask = MemoryManagement::SharedPointer<Image2D>(m_mask());
        }

        return success;
    }

    /******************************************************************************
     *  CloneSelf
     ******************************************************************************/
    Effect2D::SharedPointer GlMaskEffect::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(GlMaskEffect)(*this));
    }

}   // namespace Candera
