//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GL_BITMAP_BRUSH_MASK_BLEND_H)
    #define CANDERA_GL_BITMAP_BRUSH_MASK_BLEND_H

#include <Candera/Engine2D/Effects/CombinedEffect2D.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>
#include <CanderaPlatform/Device/Common/Effects/GlMaskEffect.h>
#include <CanderaPlatform/Device/Common/Effects/BlendEffect.h>

namespace Candera {

    class Image2D;

    /** 
     *  @brief  Output a bitmap image, apply an alpha mask, and blend it with the store buffer.
     *          Same as chaining (BitmapBrush + GlMaskEffect + BlendEffect)
     *          @see Candera::BitmapBrush
     *          @see Candera::GlMaskEffect
     *          @see Candera::BlendEffect
     */
    class GlBitmapBrushMaskBlend : public CombinedEffect2D {

        FEATSTD_TYPEDEF_BASE(CombinedEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<GlBitmapBrushMaskBlend> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class
         *  @return A pointer to the created object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~GlBitmapBrushMaskBlend();

        /**
         *  Gets a valid BrushEffect2D pointer if the effect is a brush effect or
         *  if it is a combined effect and has a brush effect, otherwise 0.
         *  @return     A valid BrushEffect2D pointer if the effect is a brush effect or
         *              if it is a combined effect and has a brush effect, otherwise 0.
         */
        virtual BrushEffect2D* GetBrushEffect2D() { return &m_bitmapBrush; }

        // overrides Effect2D::GetInplaceEffect2DCount
        virtual UInt8 GetInPlaceEffect2DCount() const { return 1; }

        // overrides Effect2D::GetInPlaceEffect2D        
        virtual InPlaceEffect2D* GetInPlaceEffect2D(UInt8 index) { return (index == 0)? &m_maskEffect : 0; }

        /**
         *  Gets a valid BlendEffect2D pointer if the effect is a blend effect or
         *  if it is a combined effect and has a blend effect, otherwise 0.
         *  @return     A valid BlendEffect2D pointer if the effect is a blend effect or
         *              if it is a combined effect and has a blend effect, otherwise 0.
         */
        virtual BlendEffect2D* GetBlendEffect2D() { return &m_blendEffect; }

        /**
         *  Retrieves the current bounding rectangle.
         *  @param boundingRectangle This value will hold the current bounding rectangle after function-call.
         */
        virtual void GetBoundingRectangle(Rectangle& boundingRectangle) const { m_bitmapBrush.GetBoundingRectangle(boundingRectangle); };

        /**
         *  Uploads all associated resources.
         *  @return true if successful.
         */
        virtual bool Upload();

        /**
         *  Unloads all associated resources.
         *  @return true if successful.
         */
        virtual bool Unload();

        /**
         *  Uploads all resources that changed since last Upload/Update.
         *  Unloads resources removed from the effect, that were previously uploaded
         *  by an Upload/Update.
         *  This should only be called on Uploaded effects. (otherwise, call Upload.)
         *  @return true if successful.
         */
        virtual bool Update();

        /**
         *  Retrieves the bitmap brush.
         *  @return     The bitmap brush.
         */
        const BitmapBrush& GetBitmapBrush() const { return m_bitmapBrush; }
        BitmapBrush& GetBitmapBrush() { return m_bitmapBrush; }

        /**
         *  Retrieves the mask effect.
         *  @return     The mask effect.
         */
        const GlMaskEffect& GetMaskEffect() const { return m_maskEffect; }
        GlMaskEffect& GetMaskEffect() { return m_maskEffect; }

        /**
         *  Retrieves the blend effect.
         *  @return     The blend effect.
         */
        const BlendEffect& GetBlendEffect() const { return m_blendEffect; }
        BlendEffect& GetBlendEffect() { return m_blendEffect; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /// @cond excluded from doxygen
        CdaEffect2DCombinedDef(Candera::GlBitmapBrushMaskBlend, GlBitmapBrushMaskBlend,  EFFECT2D_TYPE_BRUSH, EFFECT2D_TYPE_BLEND)
            CdaEffect2DProperties()

                CdaEffect2DImportProperties(BitmapBrush, Candera::BitmapBrush, m_bitmapBrush)
                CdaEffect2DImportProperties(GlMaskEffect, Candera::GlMaskEffect, m_maskEffect)
                CdaEffect2DImportProperties(BlendEffect, Candera::BlendEffect, m_blendEffect)

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        GlBitmapBrushMaskBlend();
        GlBitmapBrushMaskBlend(const GlBitmapBrushMaskBlend& rhs);
        GlBitmapBrushMaskBlend& operator = (const GlBitmapBrushMaskBlend& rhs);

        /**
         *  Triggers the rendering for the effect.
         *  @param  input       Defines the input for the effect. The first effect inside the effect list NO input. The input for the second
         *                      effect in the list must be set to the output of the first effect and so on.
         *  @param  inputArea   Area of the input that should be used as input. This should by supplied through the ouputArea parameter of the last effect.
         *  @param  transform   Defines the transformation matrix for the effect.
         *  @param  node        Node this effect is referenced to, containing transformation matrix for the effect.
         *  @param  output      Defines the output for the effect. The last effect inside the effect list must be the context of the render target.
         *  @param  outputArea  Output parameter through which the effect supplies the area of the output that was modified.
         */
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        // Use class wrapper to forward protected members.
        class BitmapBrushWrapper : public BitmapBrush {
            public:
                using BitmapBrush::Render;
        } m_bitmapBrush;
        class MaskEffectWrapper : public GlMaskEffect {} m_maskEffect;
        class BlendEffectWrapper : public BlendEffect {} m_blendEffect;
    };


}   // namespace Candera

#endif  // CANDERA_GL_BITMAP_BRUSH_MASK_BLEND_H
