//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ColorEffect.h"
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>

namespace Candera {

    FEATSTD_RTTI_DEFINITION(ColorEffect, InPlaceEffect2D)

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    ColorEffect::ColorEffect() :
        Base(),
        m_color(Candera::Color(1.0F, 1.0F, 1.0F, 1.0F))
    {
    }

    ColorEffect::ColorEffect(const ColorEffect& rhs) :
        Base(rhs),
        m_color(rhs.m_color.Get())
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    ColorEffect::~ColorEffect()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    ColorEffect::SharedPointer ColorEffect::Create()
    {
        CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, CANDERA_LINT_REASON_SHAREDPOINTER)

        ColorEffect* brush = FEATSTD_NEW(ColorEffect);
        FEATSTD_DEBUG_ASSERT(brush != 0);

        ColorEffect::SharedPointer sharedPtr(brush);
        return sharedPtr;
    }

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void ColorEffect::Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node,
                                     ContextHandle2D output, Rectangle& outputArea)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderEffect2D, "ColorEffect"));

        bool result = Renderer2D::SetTransformationMatrix(output, RenderDevice2D::SourceSurface, transform);
        result = result && RenderDevice2D::SetSurface(output, RenderDevice2D::SourceSurface, input);

        result = result && RenderDevice2D::SetActiveArea(
            output,
            RenderDevice2D::SourceSurface,
            inputArea.GetLeft(),
            inputArea.GetTop(),
            inputArea.GetWidth(),
            inputArea.GetHeight());

        Color::Data& color = m_color.Get();
        Float cameraAlpha = 1.0F;
        const Camera2D* camera2D = Renderer2D::GetActiveCamera();
        if (0 != camera2D) {
            if (camera2D->IsCameraEffectiveAlphaEnabled()) {
                cameraAlpha = camera2D->GetEffectiveAlphaValue();
            }
        }
        result = result && RenderDevice2D::SetSurfaceConstColor(
            output,
            RenderDevice2D::SourceSurface,
            color.red,
            color.green,
            color.blue,
            color.alpha * node.GetEffectiveAlphaValue() * cameraAlpha);

        result = result && Renderer2D::Blit(output);

        result = result && RenderDevice2D::SetSurfaceConstColor(output, RenderDevice2D::SourceSurface, 1.F, 1.F, 1.F, 1.F);

        if (result) {
            Float left;
            Float top;
            Float width;
            Float height;
            static_cast<void>(RenderDevice2D::GetUpdatedArea(output, &left, &top, &width, &height));
            outputArea = Rectangle(left, top, width, height);
        }
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Effect2D::SharedPointer ColorEffect::Clone() const
    {
        return Effect2D::SharedPointer(CANDERA_NEW(ColorEffect)(*this));
    }

}   // namespace Candera
