//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BLUR_BITMAP_BRUSH_BLEND_H)
    #define CANDERA_BLUR_BITMAP_BRUSH_BLEND_H

#include <Candera/Engine2D/Effects/CombinedEffect2D.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>
#include <CanderaPlatform/Device/Common/Effects/BlurEffect.h>
#include <CanderaPlatform/Device/Common/Effects/BlendEffect.h>

namespace Candera {

    class Image2D;

    /**
     *  @brief  Output a blurred and alpha blended bitmap image.
     *          The blur effect is created by applying a gaussean filter convolution.
     *          When blurring the image is blitted n + m times, where n and m are the dimensions
     *          of the gaussean filter kernel. On each of these rendered image a fixed alpha value,
     *          defined by the precalculated gaussean filter kernel, is applied. The
     *          filter kernel is calculated whenever the dimensions of the kernel change.
     *          The bigger the gaussean kernel is, the more the image is blurred. Of course,
     *          a bigger filter kernel results in decreased performance.
     *          @see Candera::BlurEffect
     */
    class BlurBitmapBrushBlend : public CombinedEffect2D {

        FEATSTD_TYPEDEF_BASE(CombinedEffect2D);

    public:
        typedef MemoryManagement::SharedPointer<BlurBitmapBrushBlend> SharedPointer;

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Creates an instance of this class.
         *  @return Pointer to the created BlurBitmapBrushBlend object.
         */
        FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

        /**
         *  Destructor
         */
        virtual ~BlurBitmapBrushBlend();

        // overrides Effect2D::GetBrushEffect2D
        virtual BrushEffect2D* GetBrushEffect2D() { return &m_bitmapBrush; }

        // overrides Effect2D::GetInplaceEffect2DCount
        virtual UInt8 GetInPlaceEffect2DCount() const { return 1; }

        // overrides Effect2D::GetInPlaceEffect2D
        virtual InPlaceEffect2D* GetInPlaceEffect2D(UInt8 index) { return (index == 0) ? &m_blurEffect : 0; }

        /**
         *  Gets a valid BlendEffect2D pointer if the effect is a blend effect or
         *  if it is a combined effect and has a blend effect, otherwise 0.
         *  @return     A valid BlendEffect2D pointer if the effect is a blend effect or
         *              if it is a combined effect and has a blend effect, otherwise 0.
         */
        virtual BlendEffect2D* GetBlendEffect2D() { return &m_blendEffect; }

        // overrides Effect2D::Upload
        virtual bool Upload();

        // overrides Effect2D::Unload
        virtual bool Unload();

        // overrides Effect2D::Update
        virtual bool Update();

        /**
         *  Retrieves the bitmap brush.
         *  @return     The bitmap brush.
         */
        const BitmapBrush& GetBitmapBrush() const { return m_bitmapBrush; }
        BitmapBrush& GetBitmapBrush() { return m_bitmapBrush; }

        /**
         *  Retrieves the blur effect.
         *  @return     The blur effect.
         */
        const BlurEffect& GetBlurEffect() const { return m_blurEffect; }
        BlurEffect& GetBlurEffect() { return m_blurEffect; }

        /**
         *  Retrieves the blend effect.
         *  @return     The blend effect.
         */
        const BlendEffect& GetBlendEffect() const { return m_blendEffect; }
        BlendEffect& GetBlendEffect() { return m_blendEffect; }

        // overrides Effect2D::Clone
        virtual Effect2D::SharedPointer Clone() const;

        /**
        * Mesh2D cannot be supported by all effects, as certain effects would create 2D vertex buffers, which would be overriden by the Mesh2D.
        * This flag helps to distinguish between effects that can be used with Mesh2D and effects that are not available.
        * @return Whether Mesh2D can be used with this effect (true) or not (false).
        */
        virtual bool IsMesh2DSupported() const override { return true; }

        /// @cond excluded from doxygen
        CdaEffect2DCombinedDef(Candera::BlurBitmapBrushBlend, BlurBitmapBrushBlend,  EFFECT2D_TYPE_BRUSH, EFFECT2D_TYPE_BLEND)
            CdaEffect2DProperties()

                CdaEffect2DImportProperties(BitmapBrush, Candera::BitmapBrush, m_bitmapBrush)
                CdaEffect2DImportProperties(BlurEffect, Candera::BlurEffect, m_blurEffect)
                CdaEffect2DImportProperties(BlendEffect, Candera::BlendEffect, m_blendEffect)

            CdaEffect2DPropertiesEnd()
        CdaEffect2DDefEnd()
        /// @endcond

        friend class BitmapBrush;

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        BlurBitmapBrushBlend();
        BlurBitmapBrushBlend(const BlurBitmapBrushBlend& rhs);
        BlurBitmapBrushBlend& operator = (const BlurBitmapBrushBlend& rhs);

        // overrides Effect2D::Render
        virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

    private:
        // Use class wrapper to forward protected members.
        class BitmapBrushWrapper : public BitmapBrush {} m_bitmapBrush;
        class BlurEffectWrapper : public BlurEffect {} m_blurEffect;
        class BlendEffectWrapper : public BlendEffect {} m_blendEffect;
    };

}   // namespace Candera

#endif  // CANDERA_BLUR_BITMAP_BRUSH_BLEND_H
