//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BLEND_EFFECT_H)
    #define CANDERA_BLEND_EFFECT_H

#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <Candera/Engine2D/Effects/BlendEffect2D.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/Engine2D/Property/BlendFactorProperty.h>
#include <Candera/Engine2D/Property/BlendOperationProperty.h>
#include <Candera/System/MetaInfo/DataType.h>
#include <CanderaPlatform/Device/Common/Effects/EffectDataTypes.h>


namespace Candera {
 
    /**
     *  @brief  Blend an image with the store buffer. The source blend factor, destination blend factor
     *          and the blend operation can be set. This can be done either combined 
     *          for the color and alpha channels or separated.
     *          @see RenderDevice2D::BlendFactor
     *          @see RenderDevice2D::BlendOperation
     *
     *  The applied operations are:
     *      Cd = Cs * Cbs cop Cd * Cbd
     *      Ad = As * Abs aop Ad * Abd
     *  where 
     *      Cd - the color written to the store buffer.
     *      Ad - the alpha value written to the store buffer.
     *      Cs - the color of the source image.
     *      As - the alpha value of the source image.
     *      Cd - the color of the destination image.
     *      Ad - the alpha value of the destination image.
     *      Cbs - the blend factor associated with the source color.
     *      Abs - the blend factor associated with the source alpha value.
     *      Cbd - the blend factor associated with the destination color.
     *      Abd - the blend factor associated with the destination alpha value.
     *      cop - the color operation.
     *      aop - the alpha operation.
     */
    class BlendEffect : public BlendEffect2D {

        FEATSTD_TYPEDEF_BASE(BlendEffect2D);

        public:

            typedef MemoryManagement::SharedPointer<BlendEffect> SharedPointer;

            FEATSTD_RTTI_DECLARATION();

            /**
             *  Creates an instance of this class
             *  @return MemoryManagement::SharedPointer to the created BlendEffect object.
             */
            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            /**
             *  Destructor
             */
            virtual ~BlendEffect();

            /**
             *  Activate the effect on a 2D context.
             *  @param output 2D context on which effect shall be activated.
             */
            void ActivateBlend(ContextHandle2D output) const;

            /**
             *  Deactivate the effect.
             *  @param output 2D context on which effect shall be deactivated.
             */
            void DeactivateBlend(ContextHandle2D output) const;

            /**
             *  Retrieves the color source blend factor property, this returned property
             *  can be changed. The performed changes will affect this class.
             *  This is the blend factor associated with the source color. (Cbs)
             *  @return     The color source blend factor property.
             */
            BlendFactorProperty& ColorSourceBlendFactor() { return m_colorBlendFactorSrc; }

            /**
             *  Retrieves the color destination blend factor property, this returned property
             *  can be changed. The performed changes will affect this class.
             *  This is the blend factor associated with the destination color. (Cbd)
             *  @return     The color destination blend factor property.
             */
            BlendFactorProperty& ColorDestinationBlendFactor() { return m_colorBlendFactorDst; }

            /**
             *  Retrieves the color blend operation property, this returned property
             *  can be changed. The performed changes will affect this class.
             *  This is color operation. (cos)
             *  @return     The color blend operation property.
             */
            BlendOperationProperty& ColorBlendOperation() { return m_colorBlendOperation; }

            /**
             *  Retrieves the alpha source blend factor property, this returned property
             *  can be changed. The performed changes will affect this class.
             *  This is the blend factor associated with the source alpha value. (Abs)
             *  @return     The alpha source blend factor property.
             */
            BlendFactorProperty& AlphaSourceBlendFactor() { return m_alphaBlendFactorSrc; }

            /**
             *  Retrieves the alpha destination blend factor property, this returned property
             *  can be changed. The performed changes will affect this class.
             *  This is the blend factor associated with the destination alpha value. (Abd)
             *  @return     The alpha destination blend factor property.
             */
            BlendFactorProperty& AlphaDestinationBlendFactor() { return m_alphaBlendFactorDst; }

            /**
             *  Retrieves the alpha blend operation property, this returned property
             *  can be changed. The performed changes will affect this class.
             *  This is alpha operation. (aos)
             *  @return     The alpha blend operation property.
             */
            BlendOperationProperty& AlphaBlendOperation() { return m_alphaBlendOperation; }

            /**
             *  Convenience method to set blend factor for source and destination and the blend operation in one single function call.
             *  The same values are set for color and alpha.
             *  @param  blendFactorSrc      The source blend factor for color and alpha.
             *  @param  blendFactorDst      The destination blend factor for color and alpha.
             *  @param  blendOperation      The blend operation for color and alpha.
             */
            void SetBlendMode(RenderDevice2D::BlendFactor blendFactorSrc, RenderDevice2D::BlendFactor blendFactorDst, RenderDevice2D::BlendOperation blendOperation);

            /**
             *  Convenience method to set blend factor for source and destination and the blend operation in one single function call.
             *  The values are set separately for color and alpha.
             *  @param  colorBlendFactorSrc      The color source blend factor.
             *  @param  colorBlendFactorDst      The color destination blend factor.
             *  @param  colorBlendOperation      The color blend operation.
             *  @param  alphaBlendFactorSrc      The alpha source blend factor.
             *  @param  alphaBlendFactorDst      The alpha destination blend factor.
             *  @param  alphaBlendOperation      The alpha blend operation.
             */
            void SetBlendModeSeparate(RenderDevice2D::BlendFactor colorBlendFactorSrc, RenderDevice2D::BlendFactor colorBlendFactorDst,
                                      RenderDevice2D::BlendOperation colorBlendOperation,
                                      RenderDevice2D::BlendFactor alphaBlendFactorSrc, RenderDevice2D::BlendFactor alphaBlendFactorDst,
                                      RenderDevice2D::BlendOperation alphaBlendOperation);

            /**
             *  Convenience method to set blend factor for source and destination and the blend operation in one single function call.
             *  @param  blendMode       The blend mode to be set.
             */
            void SetBlendMode(const RenderDevice2D::BlendMode& blendMode);

            /**
             *  Retrieves the blend mode.
             *  @return         The currently set blend mode.
             */
            RenderDevice2D::BlendMode GetBlendMode() const;

            // overrides Effect2D::Clone
            virtual Effect2D::SharedPointer Clone() const;

            /// @cond excluded from doxygen
            CdaEffect2DDef(Candera::BlendEffect, BlendEffect, EFFECT2D_TYPE_BLEND)
                CdaEffect2DProperties()

                    CdaEffect2DProperty(ColorBlendFactorSrc, RenderDevice2D::BlendFactor, m_colorBlendFactorSrc)
                        CdaDescription("The color source blend factor.")
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(ColorBlendFactorDst, RenderDevice2D::BlendFactor, m_colorBlendFactorDst)
                        CdaDescription("The color destination blend factor.")
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(ColorBlendOperation, RenderDevice2D::BlendOperation, m_colorBlendOperation)
                        CdaDescription("The color blend operation.")
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(AlphaBlendFactorSrc, RenderDevice2D::BlendFactor, m_alphaBlendFactorSrc)
                        CdaDescription("The alpha source blend factor.")
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(AlphaBlendFactorDst, RenderDevice2D::BlendFactor, m_alphaBlendFactorDst)
                        CdaDescription("The alpha destination blend factor.")
                    CdaEffect2DPropertyEnd()

                    CdaEffect2DProperty(AlphaBlendOperation, RenderDevice2D::BlendOperation, m_alphaBlendOperation)
                        CdaDescription("The alpha blend operation.")
                    CdaEffect2DPropertyEnd()

                CdaEffect2DPropertiesEnd()
            CdaEffect2DDefEnd()
            /// @endcond

        protected:
            // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
            BlendEffect();
            BlendEffect(const BlendEffect& rhs);
            BlendEffect& operator = (const BlendEffect& rhs);

            // overrides Effect2D::Render
            virtual void Render(SurfaceHandle input, const Rectangle& inputArea, const Matrix3x2& transform, const Node2D& node, ContextHandle2D output, Rectangle& outputArea);

        private:
            BlendFactorProperty m_colorBlendFactorSrc;
            BlendFactorProperty m_colorBlendFactorDst;
            BlendOperationProperty m_colorBlendOperation;
            BlendFactorProperty m_alphaBlendFactorSrc;
            BlendFactorProperty m_alphaBlendFactorDst;
            BlendOperationProperty m_alphaBlendOperation;
    };

}   // namespace Candera

#endif  // CANDERA_BLEND_EFFECT_H
